# -*- coding: utf-8 -*-
import json
import logging
import shlex
import subprocess
import os
import time

from sandbox import sdk2
from sandbox.sandboxsdk.environments import SvnEnvironment
from sandbox.sdk2 import svn


from sandbox.projects.EntitySearch import resource_types
from sandbox.projects.common.nanny import nanny
from sandbox.projects.common.search import bugbanner2
from sandbox.projects.common import string

from sandbox.projects.EntitySearch.common.current_production import get_resource_task_id_from_service


# TODO: change for better way finding production resources
PRODUCTION_ENTITYSEARCH_NANNY_SERVICE = 'sas-production-entitysearch-yp'


class EntitySearchFreshBuild(nanny.ReleaseToNannyTask2, bugbanner2.BugBannerTask):
    """ Сборка фреша EntitySearch """

    class Requirements(sdk2.Task.Requirements):
        disk_space = 8 * 1024  # 8 GB
        cores = 1
        environments = (SvnEnvironment(), )

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        fresh_url = sdk2.parameters.String(
            'Fresh url',
            default='svn+ssh://arcadia.yandex.ru/robots/trunk/wizard-data/entity_search/fresh',
        )

        fresh_revision = sdk2.parameters.Integer(
            'Fresh revision to build',
            default=0,
        )

        patch = sdk2.parameters.String(
            'Fresh patch',
            description="Text diff: applied with '-p0' inside 'fresh' dir",
            default='',
            multiline=True,
        )

        # TODO: change for dict
        fresh_resource_attrs = sdk2.parameters.String(
            'Fresh resource attributes',
            description='version attributes of sb resources included in fresh',
            default='',
        )

        sticky_gzts = sdk2.parameters.List(
            'sticky gazetters',
            description="Paths to sticky gazetteer files (relative to svn 'fresh/sticky' dir) to be compiled",
            default=[
                'index.gzt',
                'realtime.gzt',
            ]
        )

    def add_version_file(self, fresh_path, fresh_revision):
        version_info = {
            'SvnRevision': fresh_revision,
            'SandboxTask': str(self.id),
            'GenerationTime': time.strftime('%d.%m.%Y %H:%M'),
            'Revision': int(fresh_revision),
            'Task': int(self.id),
        }
        with open(os.path.join(fresh_path, 'version.info'), 'w') as h:
            json.dump(version_info, h)

    @staticmethod
    def get_production_gzt_compiler():
        token = sdk2.Vault.data('robot-ontodb', 'nanny-oauth-token')
        es_binaries_build_task_id = get_resource_task_id_from_service(
            PRODUCTION_ENTITYSEARCH_NANNY_SERVICE,
            resource_types.ENTITY_SEARCH_EXECUTABLE,
            token,
        )

        es_gzt_compiler_found = sdk2.Resource.find(
            task_id=es_binaries_build_task_id,
            type=resource_types.ENTITY_SEARCH_GZT_COMPILER_TOOL
        ).first()
        logging.info('Using gzt compiler {}'.format(es_gzt_compiler_found.id))
        return es_gzt_compiler_found

    def compile_gzt_file(self, es_gzt_compiler_path, src_path, dst_path=None, logger_prefix=None):
        if not dst_path:
            dst_path = src_path + '.bin'
        cmd = '{bin} -B {input} {output}'.format(bin=es_gzt_compiler_path, input=src_path, output=dst_path)

        logger_prefix = logger_prefix or 'compile_' + src_path.replace('/', '_')
        with sdk2.helpers.ProcessLog(self, logger=logger_prefix) as pl:
            subprocess.check_call(shlex.split(cmd), stdout=pl.stdout, stderr=pl.stderr)

    def compile_gazetteers(self, gzts_dir, es_gzt_compiler_path, sticky_gzts, sticky_gzts_to_remove):
        fresh_sticky_gzts = frozenset([os.path.join('sticky', sg) for sg in sticky_gzts])
        for root, _, files in os.walk(gzts_dir):
            fresh_path = os.path.relpath(root, gzts_dir)
            if fresh_path.startswith('sticky'):
                sticky_gzt_files = [f for f in files if f.endswith('.gzt')]
                files = [sg for sg in sticky_gzt_files if os.path.join(fresh_path, sg) in fresh_sticky_gzts]
                sticky_gzts_to_remove.extend(os.path.join(root, f) for f in sticky_gzt_files)
            for f in files:
                if f.endswith('.gzt'):
                    self.compile_gzt_file(
                        es_gzt_compiler_path,
                        src_path=os.path.join(root, f),
                        logger_prefix='compile_' + f.replace('/', '_'),
                    )

    def on_execute(self):
        self.add_bugbanner(bugbanner2.Banners.EntitySearch, component='fresh_releases')

        fresh_revision = self.Parameters.fresh_revision
        if not fresh_revision:
            fresh_revision = svn.Arcadia.info(self.Parameters.fresh_url)['commit_revision']
            logging.info('Set fresh revision to {}'.format(fresh_revision))

        fresh_attributes = {'revision': fresh_revision}

        fresh_path = svn.Arcadia.export(self.Parameters.fresh_url, path='fresh', revision=fresh_revision)

        if self.Parameters.patch:
            patch_path = os.path.join(fresh_path, 'patch.txt')
            with open(patch_path, 'wb') as patch_file:
                patch_file.write(self.Parameters.patch.encode('utf8'))

            cmd = 'patch -p0 -i {patch} -d {dir}'.format(patch=patch_path, dir=fresh_path)
            with sdk2.helpers.ProcessLog(self, logger='patch_fresh') as pl:
                subprocess.check_call(shlex.split(cmd), stdout=pl.stdout, stderr=pl.stderr)

            fresh_attributes['patched'] = True

        es_gzt_compiler_resource = sdk2.ResourceData(self.get_production_gzt_compiler())
        es_gzt_compiler_path = str(es_gzt_compiler_resource.path.absolute())
        sticky_gazetteers_to_remove = []  # list of text .gzt files in sticky folder
        self.compile_gazetteers(
            fresh_path, es_gzt_compiler_path, self.Parameters.sticky_gzts, sticky_gazetteers_to_remove
        )

        for s in sticky_gazetteers_to_remove:
            os.remove(s)

        self.add_version_file(fresh_path, fresh_revision)
        if self.Parameters.fresh_resource_attrs:
            input_resources_attr = string.parse_attrs(self.Parameters.fresh_resource_attrs)
            fresh_attributes.update(input_resources_attr)
            self.Context.resource_attributes = fresh_attributes

        fresh_resource = resource_types.ENTITY_SEARCH_FRESH(
            self,
            'entitysearch fresh',
            fresh_path,
            ttl=14,
            **fresh_attributes
        )

        fresh_resource = sdk2.ResourceData(fresh_resource)
        fresh_resource.ready()
