# -*- coding: utf-8 -*-
import json
import logging
import os

import sandbox.common.types.task as ctt
from sandbox import sdk2
from sandbox.common import errors
from sandbox.sandboxsdk.environments import SvnEnvironment
from sandbox.sdk2 import svn


import sandbox.projects.EntitySearch.fresh_build as fresh_build_utils
from sandbox.projects.EntitySearch.main_delta_resource_preparation import SELF_RELEASED_COMMIT_MESSAGE_PREFIX
from sandbox.projects.EntitySearch.common import check_task_status_succeed
from sandbox.projects.EntitySearch import resource_types
import sandbox.projects.EntitySearchShooter as shooter_utils
from sandbox.projects.common.BaseTestTask import BaseDolbiloTask as dolbilo_utils

from sandbox.projects.common.wizard.providers import entitysearch_provider as es_provider

from sandbox.projects.EntitySearch.common.current_production import (
    get_current_service_sandbox_files,
    get_current_production_resource_id_from_service,
    get_resource_id_from_service_sandbox_files
)


# TODO: change for better way finding production resources
PRODUCTION_ENTITYSEARCH_NANNY_SERVICE = 'sas-production-entitysearch-yp'


def are_changes_self_released(commit):
    commit_msg = commit['msg'].strip()
    return commit_msg.startswith(SELF_RELEASED_COMMIT_MESSAGE_PREFIX)


def format_commits(commits):
    return u'\n'.join(
        u'r{revision} by {author} ({date}): {msg}'.format(
            revision=commit['revision'],
            author=commit['author'],
            date=commit['date'],
            msg=commit['msg'].strip()
        ) for commit in commits
    ).encode('utf-8')


COMMIT_REVISION = 'commit_revision'
DATE = 'date'
FRESH_SVN_PATH = 'svn+ssh://arcadia.yandex.ru/robots'
FRESH_DIR = '/trunk/wizard-data/entity_search/fresh'
FRESH_URL = FRESH_SVN_PATH + FRESH_DIR


# TODO: add bugbunny task (?) - check release machine integration
class EntitySearchFreshPreparation(sdk2.Task):
    """
        Автоматические подготовка и релиз фреша (sdk2)
    """

    class Requirements(sdk2.Task.Requirements):
        disk_space = 4096  # 4 Gb
        cores = 1
        environments = (SvnEnvironment(), )

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        debug_mode = sdk2.parameters.Bool(
            'Debug mode',
            description="Defines fresh resource release type (False for 'stable') + forces fresh building if True",
            default=True
        )

        force_shooting = sdk2.parameters.Bool(
            'Force shooting',
            description="Do shooting anyway (mostly debug option)",
            default=False,
        )

        no_test_paths = sdk2.parameters.List(
            'No test paths',
            description="Paths to files (relative to svn 'fresh' dir) that don't need testing (ENTITYSEARCH_SHOOTER)",
            default=[
                'auto_db.trie',
                'auto_index.gzt.bin',
                'fixlist.txt',
                'main_delta.gzt.bin',
                'main_delta.trie',
                'minibase_market_tr.gzt.bin',
                'minibase_market_tr.trie',
                'sticky/fixlist.gzt',
            ]
        )

        self_released_commits_no_build_limit = sdk2.parameters.Integer(
            'Self released commits no build limit',
            description=(
                "Task doesn't build fresh if commits are only self-released and their amount is lower than this number"
            ),
            default=500  # > 6 deltas/hour * 24 hours/day * 3 days = 432 deltas
        )

    @property
    def footer(self):
        if self.Context.unreleased_commits:
            return '<b>Found commits</b>:<br/>' + format_commits(self.Context.unreleased_commits).replace('\n', '<br/>')

    def run_shooting(self):
        token = sdk2.Vault.data('robot-ontodb', 'nanny-oauth-token')
        sandbox_files = get_current_service_sandbox_files(PRODUCTION_ENTITYSEARCH_NANNY_SERVICE, token)
        dumped_sb_files = json.dumps(sandbox_files, indent=4)
        logging.info("%s service's sandbox files:\n%s", PRODUCTION_ENTITYSEARCH_NANNY_SERVICE, dumped_sb_files)

        def current_resource(resource_type):
            return get_resource_id_from_service_sandbox_files(resource_type, sandbox_files)

        built_fresh = sdk2.Resource.find(
            type=resource_types.ENTITY_SEARCH_FRESH,
            task_id=self.Context.fresh_build_task
        ).first()

        if not built_fresh:
            err_msg = "Can't find resource of type {} with task id {} (expected built fresh!)".format(
                resource_types.ENTITY_SEARCH_FRESH,
                self.Context.fresh_build_task
            )
            raise errors.TaskFailure(err_msg)

        shooter_task_class = sdk2.Task[shooter_utils.EntitySearchShooter.type]

        input_parameters = {
            es_provider.EntitySearchBinary.name: current_resource(resource_types.ENTITY_SEARCH_EXECUTABLE),
            es_provider.EntitySearchData.name: current_resource(resource_types.ENTITY_SEARCH_DATA),
            es_provider.EntitySearchConfig.name: current_resource(resource_types.ENTITY_SEARCH_CONFIG),
            es_provider.Fresh.name: built_fresh.id,
            es_provider.NerData.name: current_resource(resource_types.ENTITY_SEARCH_NER_DATA),
            dolbilo_utils.RequestsLimit.name: 30000,
            dolbilo_utils.TotalSessions.name: 1,
            shooter_utils.CheckQuality.name: True,
        }

        # TODO: add new preparing logic to Shooter (after OBJECTS-8904)
        shooter_task = shooter_task_class(
            self,
            description='Test shooting on fresh release (rev. {})'.format(
                self.Context.current_fresh_svn_info[COMMIT_REVISION]
            ),
            fail_on_any_error=False,
            **input_parameters
        )

        logging.info('Running shooting task')
        shooter_task.enqueue()
        return shooter_task.id

    def fresh_resource_in_production(self):
        token = sdk2.Vault.data('robot-ontodb', 'nanny-oauth-token')
        fresh_resource_id = get_current_production_resource_id_from_service(
            PRODUCTION_ENTITYSEARCH_NANNY_SERVICE,
            resource_types.ENTITY_SEARCH_FRESH,
            token
        )

        logging.info('Production fresh resource id: {}'.format(fresh_resource_id))
        fresh_resource = sdk2.Resource.find(type=resource_types.ENTITY_SEARCH_FRESH, id=fresh_resource_id).first()
        if not fresh_resource:
            err_msg = "Can't find resource of type {} with id {}".format(
                resource_types.ENTITY_SEARCH_FRESH,
                fresh_resource_id
            )
            raise errors.TaskFailure(err_msg)
        return fresh_resource

    def fresh_revision_in_production(self):
        fresh_resource = self.fresh_resource_in_production()
        revision = fresh_resource.revision
        logging.info('Fresh revision in production: {}'.format(revision))
        return revision

    def need_to_build_fresh(self):
        released_fresh_rev = int(self.Context.previous_release_revision)
        new_fresh_rev = int(self.Context.current_fresh_svn_info[COMMIT_REVISION])

        logging.info(
            'Production fresh revision: {}. Svn fresh commit revision: {}'.format(released_fresh_rev, new_fresh_rev)
        )

        if released_fresh_rev >= new_fresh_rev:
            logging.info('New data was not found')
            return False

        unreleased_commits = filter(lambda commit: not are_changes_self_released(commit), self.Context.commits)
        if unreleased_commits:
            self.Context.unreleased_commits = sorted(
                unreleased_commits,
                key=lambda commit: int(commit['revision']),
                reverse=True
            )
            logging.info('Found commits:\n%s', format_commits(self.Context.unreleased_commits))
            return True

        if len(self.Context.commits) >= self.Parameters.self_released_commits_no_build_limit:
            logging.info('Fresh is too outdated, updating')
            return True

        logging.info('All commits ({}) were self released, no need to build fresh'.format(len(self.Context.commits)))
        return False

    def make_release(self):
        logging.info('Creating fresh release')
        release_subj = '''
            Fresh release (rev. {rev}, mod. date {mod_date}). Task: https://sandbox.yandex-team.ru/task/{id}/view
        '''.strip()

        release_params = {
            'task_id': int(self.Context.fresh_build_task),
            'type': 'testing' if self.Parameters.debug_mode else 'stable',
            'subject': release_subj.format(
                rev=self.Context.current_fresh_svn_info[COMMIT_REVISION],
                mod_date=self.Context.current_fresh_svn_info[DATE],
                id=self.id
            ),
            'to': ['entity-search-releases'],
            'cc': [self.author],
            'params': {},
            'message': '',
        }

        if self.Context.unreleased_commits:
            release_params['message'] = 'Released commits:\n' + format_commits(self.Context.unreleased_commits)

            if not self.Parameters.debug_mode:
                release_params['cc'].extend(commit['author'] for commit in self.Context.unreleased_commits)

        self.server.release(release_params)
        logging.info('Fresh release created')

    def commits_since_current_fresh_release(self):
        if self.Context.previous_release_revision == self.Context.current_fresh_svn_info[COMMIT_REVISION]:
            return []

        commits = svn.Svn.log(
            url=FRESH_URL,
            revision_from=int(self.Context.previous_release_revision) + 1,
            revision_to=int(self.Context.current_fresh_svn_info[COMMIT_REVISION])
        )

        for commit in commits:
            commit['date'] = commit['date'].strftime('%Y-%m-%dT%H:%M:%S.%fZ')

        return commits

    def set_svn_info(self):
        self.Context.previous_release_revision = self.fresh_revision_in_production()

        svn_info = svn.Svn.info(FRESH_URL)
        logging.info(
            'Fresh svn info:\n' + '\n'.join(
                '{}: {}'.format(k, v) for k, v in sorted(svn_info.iteritems())
            )
        )

        self.Context.current_fresh_svn_info = svn_info
        self.Context.commits = self.commits_since_current_fresh_release()

    @staticmethod
    def need_testing(commits, no_test_paths):
        for commit in commits:
            for path in commit['paths']:
                action, fs_path = path
                # Test if any file is not modified, but added or deleted
                filename = os.path.relpath(fs_path, FRESH_DIR)
                if action != 'M':
                    logging.info("Found '{}\t{}'. Test shooting will be executed".format(action, filename))
                    return True
                if filename not in no_test_paths:
                    logging.info('Modified file was found: {}. Test shooting will be executed'.format(filename))
                    return True

        logging.info('There is no need to run shooting. Only auto files were modified')
        return False

    def init_fresh_building(self, fresh_revision):
        fresh_build_task = fresh_build_utils.EntitySearchFreshBuild(
            self,
            notifications=self.Parameters.notifications
        )
        fresh_build_task.Parameters.fresh_revision = fresh_revision
        fresh_build_task.enqueue()
        return fresh_build_task.id

    def on_execute(self):
        with self.memoize_stage.init_context(commit_on_entrance=False):
            self.set_svn_info()
            if not self.need_to_build_fresh() and not self.Parameters.debug_mode:
                return

        with self.memoize_stage.do_fresh_building():
            self.Context.fresh_build_task = self.init_fresh_building(
                self.Context.current_fresh_svn_info[COMMIT_REVISION]
            )
            raise sdk2.WaitTask(self.Context.fresh_build_task, ctt.Status.Group.FINISH | ctt.Status.Group.BREAK)

        check_task_status_succeed(task_id=self.Context.fresh_build_task, raise_not_ok=True)

        with self.memoize_stage.do_shooting_if_necessary():
            if self.need_testing(self.Context.commits, self.Parameters.no_test_paths) or self.Parameters.force_shooting:
                self.Context.shooting_task = self.run_shooting()
                raise sdk2.WaitTask(self.Context.shooting_task, ctt.Status.Group.FINISH | ctt.Status.Group.BREAK)

        if self.Context.shooting_task:
            check_task_status_succeed(task_id=self.Context.shooting_task, raise_not_ok=True)

        self.make_release()
