from tarfile import TarFile

import sandbox.common.types.task as ctt
from sandbox import sdk2
from sandbox.common import errors
from sandbox.sdk2 import svn, ssh
from sandbox.sdk2.path import Path

import sandbox.projects.EntitySearchConvertDb as convert_task_utils
from sandbox.projects.common.nanny import nanny
from sandbox.projects.EntitySearch import resource_types
from sandbox.projects.EntitySearch.common import check_task_status_succeed


SELF_RELEASED_COMMIT_MESSAGE_PREFIX = '[SELF_RELEASED]'


def make_task_url(task_id):
    return 'https://sandbox.yandex-team.ru/task/{}/view'.format(task_id)


class EntitySearchMainDeltaResourcePreparation(nanny.ReleaseToNannyTask2, sdk2.Task):
    ARCADIA_FRESH_URL = 'svn+ssh://arcadia.yandex.ru/robots/trunk/wizard-data/entity_search/fresh'
    FRESH_LOCAL_PATH = 'fresh'
    EXPECTED_CONVERTOR_RESOURCE_FILE_SUFFIXES = ('.trie.tar', '.gzt.bin.tar')

    class Requirements(sdk2.Task.Requirements):
        disk_space = 8192  # 8 Gb
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.Group('Main delta parameters for convertor'):
            yt_server_name = sdk2.parameters.String('YT server FQDN', default='hahn.yt.yandex.net', required=True)
            input_table = sdk2.parameters.String('YT path to main delta table', required=True)

        description = 'Trie + Gzt from YT table ({})'.format(input_table)

        db_type = sdk2.parameters.String(
            'Database type',
            description='Delta local path and expected prefix for convertor resource files',
            default='main_delta',
            required=True
        )

        delta_resource_type = sdk2.parameters.String(
            'Delta resource type',
            description='Output sandbox resource type',
            default='ENTITY_SEARCH_MAIN_DELTA',
            required=True
        )

        # TODO: remove after testing
        convertor_task = sdk2.parameters.Integer(
            'Existing convertor task', description='Independent convertor task which resources will be used', default=0
        )

        resource_attributes = sdk2.parameters.Dict(
            'Resource attributes',
            description='Attributes to assign to the result resource (i.e. main delta version)',
            default={}
        )

        debug_mode = sdk2.parameters.Bool(
            "Debug mode",
            description="No effects atm, for back compatibility",
            default=True,
            required=True
        )

        # OBJECTS-13424
        commit_to_robots_repo = sdk2.parameters.Bool(
            "Do commit to repo",
            description="Commits delta to robots' trunk repository",
            default=False
        )

        fail_on_any_error = True

    def init_table_converting(self):
        convertor_task_class = sdk2.Task[convert_task_utils.EntitySearchConvertDb.type]
        convertor_task_id = self.Parameters.convertor_task

        if convertor_task_id:
            fetched_convertor_task = sdk2.Task.find(
                task_type=convertor_task_class,
                task_id=convertor_task_id
            ).first()
            if not fetched_convertor_task:
                err_msg = 'Task {} is not valid (must have {} type)'
                raise errors.TaskFailure(err_msg.format(convertor_task_id, convertor_task_class.type))
            self.Context.convertor_task = convertor_task_id
        else:
            input_parameters = {
                convert_task_utils.ServerName.name: self.Parameters.yt_server_name,
                convert_task_utils.Table.name: self.Parameters.input_table,
                convert_task_utils.ResultPrefix.name: self.Parameters.db_type,
                convert_task_utils.CodecType.name: convert_task_utils.NO_CODEC,
            }
            requirements = {
                'disk_space': 8 << 30,  # 8 Gb
                'ram': 4 << 30,         # 4 Gb
            }

            convertor_task = convertor_task_class(
                self,
                notifications=self.Parameters.notifications,
                **input_parameters
            )
            sdk2.Task.server.task[convertor_task.id].update({'requirements': requirements})
            convertor_task.enqueue()

            self.Context.convertor_task = convertor_task.id

    @staticmethod
    def unpack_resources(resources, dst):
        for resource in resources:
            with TarFile(str(resource.path)) as t:
                t.extractall(dst)

    def commit_to_arcadia_fresh(self, resources, fresh_local_path):
        self.unpack_resources(resources, dst=fresh_local_path)

        with ssh.Key(self, 'robot-ontodb', 'robot-ontodb-ssh-key'):
            formatted_attributes_string = ', '.join(
                '{}={}'.format(k, v) for k, v in self.Parameters.resource_attributes.iteritems()
            )
            commit_msg = '\n'.join(
                (
                    SELF_RELEASED_COMMIT_MESSAGE_PREFIX + ' New main delta ({}) by {}.'.format(
                        formatted_attributes_string, self.author
                    ),
                    'Task: {}'.format(make_task_url(self.id)),
                )
            )

            svn.Arcadia.commit(fresh_local_path, commit_msg, user='robot-ontodb')

    def make_delta_resource(self, resources):
        main_delta_resource = sdk2.Resource[self.Parameters.delta_resource_type](
            task=self,
            description='Delta for table {}'.format(self.Parameters.input_table),
            path=self.Parameters.db_type,
            **self.Parameters.resource_attributes
        )

        main_delta_resource_data = sdk2.ResourceData(main_delta_resource)
        main_delta_resource_data.path.mkdir(exist_ok=True)

        self.unpack_resources(resources, dst=str(main_delta_resource_data.path))
        main_delta_resource_data.ready()

    def sync_converted_resources(self):
        converted_resources = (
            sdk2.Resource.find(
                task_id=self.Context.convertor_task,
                type=resource_types.ENTITY_SEARCH_GAZETEER,
            ).first(),
            sdk2.Resource.find(
                task_id=self.Context.convertor_task,
                type=resource_types.ENTITY_SEARCH_TRIE,
                attrs={'db_type': self.Parameters.db_type}
            ).first(),
        )

        packed_resources = []
        for resource in converted_resources:
            packed_resources.append(sdk2.ResourceData(resource))

        expected_convertor_resource_files = {self.Parameters.db_type + suffix for suffix in self.EXPECTED_CONVERTOR_RESOURCE_FILE_SUFFIXES}
        found_resource_files = set(
            resource.path.name for resource in packed_resources
        )
        if found_resource_files != expected_convertor_resource_files:
            err_msg = "Can't find necessary resources in convertor task ({})\nexpected {}\nfound {}"
            raise errors.TaskFailure(
                err_msg.format(
                    make_task_url(self.Context.convertor_task),
                    ", ".join(expected_convertor_resource_files),
                    ", ".join(found_resource_files),
                )
            )

        return packed_resources

    def on_execute(self):
        with self.memoize_stage.init_converting():
            self.init_table_converting()
            raise sdk2.WaitTask(self.Context.convertor_task, ctt.Status.Group.FINISH | ctt.Status.Group.BREAK)

        check_task_status_succeed(task_id=self.Context.convertor_task, raise_not_ok=True)

        local_resources = self.sync_converted_resources()

        if self.Parameters.commit_to_robots_repo:
            Path(self.FRESH_LOCAL_PATH).mkdir(exist_ok=True)
            svn.Arcadia.checkout(self.ARCADIA_FRESH_URL, self.FRESH_LOCAL_PATH)
            self.commit_to_arcadia_fresh(local_resources, self.FRESH_LOCAL_PATH)

        self.make_delta_resource(local_resources)
