# -*- coding: utf-8 -*-
import json
import logging
import os
import requests
import shutil
import subprocess
import time

from sandbox import sdk2
from sandbox.sandboxsdk import environments
from sandbox.sdk2.vcs.svn import Arcadia
from sandbox.projects.common.nanny import nanny
from sandbox.projects.EntitySearch import resource_types


def checkout_kvsaas_config(checkout_arcadia_from_url, arcadia_path, output_path):
    arcadia_url = Arcadia.parse_url(checkout_arcadia_from_url)
    config_url = arcadia_url._replace(subpath=arcadia_path)
    url = os.path.join(Arcadia.ARCADIA_SCHEME + ':', config_url.path, config_url.subpath)
    revision = config_url.revision

    logging.info('Checkout kvsaas config from: {url}, revision: {revision}'.format(url=url, revision=revision))

    with open(output_path, 'w') as config_file:
        config_file.writelines(Arcadia.cat(url, revision=revision))


def edit_kvsaas_config(config_path, **params):
    logging.info('Applying params to kvsaas config')

    if os.path.getsize(config_path) < 1:
        raise Exception('Failed to modify kvsaas config: empty original file at {}'.format(config_path))

    config_lines = []
    with open(config_path, 'r') as config_file:
        for line in config_file.readlines():
            if ':' not in line:
                continue
            key, value = line.split(':', 1)
            comment = ''
            if '#' in value:
                value, comment = value.split('#', 1)

            key, value, comment = [token.strip() for token in [key, value, comment]]

            if key in params:
                value = str(params[key])
                if isinstance(params[key], str):
                    value = '"' + value + '"'

            changed_line = key + ': ' + value
            if comment:
                changed_line += ' # ' + comment

            config_lines.append(changed_line + '\n')
    logging.info('Modified kvsaas config:\n{config}'.format(config=''.join(config_lines)))
    with open(config_path, 'w') as config_file:
        config_file.writelines(config_lines)


def send_to_ferryman(server, cluster, table, namespace):
    tables = [{
        'Cluster': cluster,
        'Path': table,
        'Namespace': namespace,
        'Timestamp': int(time.time() * 1e6),
        'Delta': False
    }]

    if not server.startswith('http://') and not server.startswith('https://'):
        server = 'http://' + server

    logging.info('Sending table {table} to ferryman server {server}, on {cluster}, to kps/namespace: {namespace}'.format(
        server=server, table=table, cluster=cluster, namespace=namespace))

    for i in range(5):
        try:
            params = {'tables': json.dumps(tables)}
            r = requests.get('{}/add-full-tables'.format(server), params=params)
            logging.info('Ferryman send request url: {url}'.format(url=r.url))
            r.raise_for_status()
            response = r.json()
            batch = response['batch']
            logging.info('Send request comleted succesfully. Ferryman batch: {}'.format(batch))
            return batch
        except Exception as e:
            logging.error('Failed to send table to ferryman. Error: {e}.'.format(e=e))
            if i >= 4:
                raise
            time.sleep(10)


def get_ferryman_batch_status(server, batch):
    if not server.startswith('http://') and not server.startswith('https://'):
        server = 'http://' + server

    for i in range(5):
        try:
            r = requests.get('{}/get-batch-status'.format(server), params={'batch': batch})
            r.raise_for_status()
            status = r.json()['status']
            logging.info('Ferryman batch status: {}'.format(status))
            return status
        except Exception as e:
            logging.error('Failed to get ferryman batch status. Error: {e}. Requested url: {url}'.format(e=e, url=r.url))
            if i >= 4:
                raise
            time.sleep(10)


class EntitySearchPrepareMainDbRelease(nanny.ReleaseToNannyTask2, sdk2.Task):
    class Requirements(sdk2.Requirements):
        environments = [environments.PipEnvironment('yandex-yt')]
        disk_space = 200 * 1024  # 200 GB

    class Parameters(sdk2.Parameters):
        checkout_arcadia_from_url = sdk2.parameters.String('Arcadia url to checkout kvsaas config template', default='svn+ssh://arcadia.yandex.ru/arc/trunk/arcadia')
        kvsaas_config_arcadia_path = sdk2.parameters.String('KvSaas config path in Arcadia', default='search/wizard/entitysearch/data/main_kvsaas.cfg')
        kvsaas_service_id = sdk2.parameters.String('KvSaas database service id', default='entitysearch_maindb')
        kvsaas_kps = sdk2.parameters.String('KvSaas kps (namespace) to release database to', required=True)
        ferryman_server = sdk2.parameters.String('Ferryman server', default='entitysearch-maindb.ferryman.n.yandex-team.ru')
        ferryman_batch = sdk2.parameters.String('Existing ferryman batch, used to prepare release when db uploading to saas was done elswhere', required=False)
        entitysearch_convert_db_archive = sdk2.parameters.Resource('Converted db resource', resource_type=resource_types.ENTITY_SEARCH_CONVERT_DB_ARCHIVE, required=True)
        kvsaas_db_artifacts_dir = sdk2.parameters.String('YT directory with prepared KvSaas database artifacts (intex table, remap.trie, list_item.trie)', required=True)
        yt_cluster = sdk2.parameters.String('YT cluster', default='hahn')
        yt_token = sdk2.parameters.YavSecret('Vault secret with YT token', default='sec-01depzzkx7y3fv804mms8qqsvy@ver-01dq5hz9v7ybxmp7wzyq6wdzs1#robot-ontodb-daily-hahn')

    class KvSaasArtifacts:
        remap_trie = 'remap.trie'
        list_item_trie = 'list_item.trie'
        index_docs = 'kvsaas_doc'

    def on_execute(self):
        import yt.wrapper as yt

        result_dir = 'main_db'

        with self.memoize_stage.send_to_ferryman(commit_on_entrance=False):
            # check resource and get db version
            entitysearch_convert_db_archive = sdk2.Resource[self.Parameters.entitysearch_convert_db_archive]
            self.Context.db_version = entitysearch_convert_db_archive.db_version
            logging.info('Converted database version: {ver}'.format(ver=self.Context.db_version))

            # upload database to kvsaas
            if self.Parameters.ferryman_batch:
                self.Context.ferryman_batch = self.Parameters.ferryman_batch
            else:
                self.Context.ferryman_batch = send_to_ferryman(
                    server=self.Parameters.ferryman_server,
                    cluster=self.Parameters.yt_cluster,
                    table=str(yt.YPath(self.Parameters.kvsaas_db_artifacts_dir).join(self.KvSaasArtifacts.index_docs)),
                    namespace=self.Parameters.kvsaas_kps
                )

        with self.memoize_stage.prepare_resource(commit_on_entrance=False):
            # copy ENTITY_SEARCH_CONVERT_DB_ARCHIVE to result dir
            entitysearch_convert_db_archive = sdk2.Resource[self.Parameters.entitysearch_convert_db_archive]
            converted_data = sdk2.ResourceData(entitysearch_convert_db_archive)
            shutil.copytree(str(converted_data.path), result_dir)
            subprocess.call(['chmod', '-R', 'a+rw', result_dir])

            kvsaas_config = os.path.join(result_dir, os.path.basename(self.Parameters.kvsaas_config_arcadia_path))
            checkout_kvsaas_config(
                self.Parameters.checkout_arcadia_from_url,
                self.Parameters.kvsaas_config_arcadia_path,
                kvsaas_config
            )
            edit_kvsaas_config(kvsaas_config, ServiceId=str(self.Parameters.kvsaas_service_id), Kps=str(self.Parameters.kvsaas_kps))

            logging.info('Downloading kvsaas db artifacts from: {cluster}.{dir}'.format(
                cluster=self.Parameters.yt_cluster,
                dir=self.Parameters.kvsaas_db_artifacts_dir))
            yt.config['proxy']['url'] = self.Parameters.yt_cluster + '.yt.yandex.net'
            yt.config['token'] = self.Parameters.yt_token.data()[self.Parameters.yt_token.default_key]
            kvsaas_db_artifacts_dir = yt.YPath(self.Parameters.kvsaas_db_artifacts_dir)

            with open(os.path.join(result_dir, 'main.remap.trie'), 'w') as remap_trie_file:
                remap_trie_file.write(yt.read_file(kvsaas_db_artifacts_dir.join(self.KvSaasArtifacts.remap_trie)).read())

            with open(os.path.join(result_dir, 'list_item.trie'), 'w') as list_item_trie_file:
                list_item_trie_file.write(yt.read_file(kvsaas_db_artifacts_dir.join(self.KvSaasArtifacts.list_item_trie)).read())

            # finalize result resource
            logging.info('Creating releasable resource')
            result_resource = resource_types.ENTITY_SEARCH_CONVERT_DB_ARCHIVE(
                task=self,
                description='Entitysearch main db release for {version}'.format(version=self.Context.db_version),
                path=result_dir,
                db_version=self.Context.db_version,
                kvsaas_service_id=str(self.Parameters.kvsaas_service_id),
                kvsaas_kps=str(self.Parameters.kvsaas_kps)
            )
            sdk2.ResourceData(result_resource).ready()
            self.Context.result_resource_id = result_resource.id

        # check database uploading status
        status = get_ferryman_batch_status(server=self.Parameters.ferryman_server, batch=self.Context.ferryman_batch)
        if status in ['queue', 'processing', 'final']:
            raise sdk2.WaitTime(10 * 60)  # wait for 10 min

        if status != 'searchable':
            logging.error('Failed to upload database to kvsaas. Status: {status}. Ferryman batch: {batch}'.format(status=status, batch=self.Context.ferryman_batch))
            # delete produced resource
            try:
                self.server.batch.resources.delete = [self.Context.result_resource_id]
            except Exception as e:
                logging.error('Failed to delete produced resource: {e}'.format(e=e))
            raise Exception('Failed to upload database to kvsaas')

        logging.info('Database uploading complete.')

    def on_release(self, additional_parameters):
        super(EntitySearchPrepareMainDbRelease, self).on_release(additional_parameters)

    @property
    def release_template(self):
        subj = 'Releasing ' + self.Parameters.description
        subj += ' (ver {})'.format(self.Context.db_version)

        return sdk2.ReleaseTemplate(
            cc=['entity-search-releases'],
            subject=subj,
            message='Released database (ver {version}) was deployed to kvsaas service: {service_id}, namespace/kps: {kps}. Ferryman batch: {batch}'.format(
                version=self.Context.db_version,
                service_id=self.Parameters.kvsaas_service_id,
                kps=self.Parameters.kvsaas_kps,
                batch=self.Context.ferryman_batch
            )
        )
