import re

from sandbox import sdk2

from sandbox.projects.common import link_builder
from sandbox.projects.common import string
from sandbox.projects.common.betas.beta_api import BetaApi

from sandbox.projects.release_machine import security as rm_sec
from sandbox.projects.release_machine import input_params2
from sandbox.projects.release_machine.components import all as release_machine_components
from sandbox.projects.release_machine.core import const as rm_const
from sandbox.projects.release_machine.core import task_env
from sandbox.projects.release_machine import yappy as yappy_helper
from sandbox.projects.release_machine.helpers import startrek_helper


class ReleaseMachineComponentedTask():
    class Parameters(sdk2.Parameters):
        with sdk2.parameters.Group('Release machine settings') as release_machine_block:
            with sdk2.parameters.String('Run mode') as run_mode_type:
                run_mode_type.values.custom = run_mode_type.Value('Custom', default=True)
                run_mode_type.values.release_machine = 'Release machine'

            with run_mode_type.value.release_machine:
                component_name_param = input_params2.ComponentName2()
                release_number = input_params2.release_number()


class BetaAcceptingTask():
    class Parameters(sdk2.Parameters):
        with sdk2.parameters.Group('Input beta settings') as beta_settings:
            beta_url = sdk2.parameters.String('url (custom or yappy beta) to run task(s) over', required=True)

            with sdk2.parameters.String('Beta source') as beta_source_type:
                beta_source_type.values.custom = beta_source_type.Value('Custom', default=True)
                beta_source_type.values.yappy_beta = 'Yappy Beta'

            with beta_source_type.value.yappy_beta:
                wait_consistency = sdk2.parameters.Bool(
                    'Wait Sample and Checked betas consistency (for Yappy betas)', default=False
                )

    class Context(sdk2.Context):
        pass

    def need_to_check_yappy_beta_status(self):
        return self.Parameters.run_mode_type == 'release_machine' and self.Parameters.wait_consistency is True

    def check_yappy_beta_status(self, beta_name):
        api_token = sdk2.Vault.data(rm_const.COMMON_TOKEN_OWNER, rm_const.YAPPY_TOKEN_NAME)
        api = BetaApi.fromurl(token=api_token)

        self.Context.waiting_beta = beta_name
        yappy_helper.wait_consistency(self, api, beta_name)
        self.Context.waiting_beta = ''


class StarTrekReportingTask(ReleaseMachineComponentedTask):
    class Requirements(sdk2.task.Requirements):
        client_tags = task_env.TaskTags.startrek_client
        environments = [task_env.TaskRequirements.startrek_client]

    STATUSES_WITH_COLORS = {
        'FAILED': 'red',
        'SUCCESS': 'green',
        'STARTED': 'blue',
    }

    @property
    def ticket_message_title(self):
        raise NotImplementedError('Define "ticket_message_title" for {}'.format(type(self)))

    @property
    def ticket_group_name(self):
        raise NotImplementedError('Define "ticket_group_name" for {} (one of rm_const.TicketGroups)'.format(type(self)))

    def _get_star_trek_helper(self):
        st_auth_token = rm_sec.get_rm_token(self)
        return startrek_helper.STHelper(st_auth_token)

    def _get_star_trek_issue(self, release_num):
        st_helper = self._get_star_trek_helper()

        component_info = release_machine_components.COMPONENTS[
            self.Parameters.component_name
        ]()
        return st_helper.find_ticket_by_release_number(release_num, component_info).key

    def _comment_star_trek_ticket(self, title, content):
        st_helper = self._get_star_trek_helper()

        component_info = release_machine_components.COMPONENTS[
            self.Parameters.component_name
        ]()

        comment, _ = st_helper.find_comment(
            release_num=self.Parameters.release_number,
            comment_start=self.ticket_group_name,
            c_info=component_info,
        )

        if not self._update_comment_text(comment, title, message=content):
            st_helper.write_grouped_comment(
                group_name=self.ticket_group_name,
                title=title,
                content=content,
                release_num=self.Parameters.release_number,
                c_info=component_info,
            )

    def _update_comment_text(self, comment, title, message):
        if not comment:
            return False

        comment_text = comment.text
        comment_parts = re.findall(r'(<{[^<]*Sandbox task:[^0-9]*([0-9]+)[^>]*}>)', comment_text)

        for text_part, task_id in comment_parts:
            if int(task_id) != self.id:
                continue

            new_text_part = '<{{{title}\n{content}}}>\n'.format(
                title=string.all_to_str(title),
                content=string.all_to_str(message)
            ).decode('utf-8')

            comment_text = comment_text.replace(text_part, new_text_part)
            comment.update(text=comment_text)

            return True

        return False

    def _make_title_suffix(self, status):
        return ' -- **!!({status_color}){status}!!**'.format(
            status_color=self.STATUSES_WITH_COLORS.get(status, 'grey'),
            status=status,
        )

    def need_to_comment_tickets(self):
        return self.Parameters.run_mode_type == 'release_machine' and self.Parameters.component_name is not None

    def add_to_star_trek_grouped_comment(self, status, content):
        """ Finds comment in ticket and task's block in it (with "Sandbox task: <id>" marker) """
        title = self.ticket_message_title + self._make_title_suffix(status)
        content_prefix = 'Sandbox task: ' + link_builder.task_wiki_link(self.id) + '\n'

        self._comment_star_trek_ticket(
            title=title,
            content=content_prefix + content,
        )
