# -*- coding: utf-8 -*-

import logging
import urllib2
import os

from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk.paths import get_logs_folder
from sandbox.sandboxsdk.process import run_process, get_process_info

from sandbox.projects import resource_types
from sandbox.projects.common.wizard.providers import EntitySearchProvider
from sandbox.projects.common.search import bugbanner

import sandbox.common.types.client as ctc


RESPONSES_FOLDER = 'responses'


class Requests(parameters.ResourceSelector):
    name = "requests"
    description = "Apphost contexts resource"
    resource_type = resource_types.ENTITY_SEARCH_APPHOST_REQUESTS
    required = True


class RequestsLimit(parameters.SandboxIntegerParameter):
    name = 'requests_limit'
    description = 'Requests limit'
    default_value = 1000


class ServantClient(parameters.ResourceSelector):
    name = 'servant_client'
    description = 'Servant client binary'
    resource_type = resource_types.ARCADIA_PROJECT_BINARY
    required = True


class EntitySearchApphostResponses(bugbanner.BugBannerTask):
    """
        Таск для вытаскивания ответов бекенда entitysearch
        по протоколу аппхоста.
    """

    type = 'ENTITYSEARCH_APPHOST_RESPONSES'

    required_ram = EntitySearchProvider.MIN_RAM
    execution_space = EntitySearchProvider.DISK_SPACE_FOR_DATA

    input_parameters = [Requests, RequestsLimit, ServantClient] + EntitySearchProvider.input_parameters
    client_tags = ctc.Tag.LINUX_PRECISE

    def on_execute(self):
        self.add_bugbanner(bugbanner.Banners.EntitySearch)

        contexts = self.sync_resource(self.ctx[Requests.name])
        servant_client = os.path.join(self.sync_resource(self.ctx[ServantClient.name]), 'servant_client')

        os.mkdir(RESPONSES_FOLDER)
        self.create_resource(RESPONSES_FOLDER, RESPONSES_FOLDER, resource_types.ENTITY_SEARCH_APPHOST_RESPONSES)

        requests_limit = self.ctx[RequestsLimit.name]
        requests_cnt = 0

        with EntitySearchProvider.from_task_context(self.ctx) as provider:
            logging.info('ps info: %s' % get_process_info(provider.process.pid, ['rss']))

            version_url = 'http://localhost:{}/?action=version'.format(provider.port)
            self.ctx['daemon_version'] = urllib2.urlopen(version_url).read()

            err = open(os.path.join(get_logs_folder(), 'servant_client.err.log'), 'w')
            for context_file in sorted(os.listdir(contexts)):
                context_path = os.path.join(contexts, context_file)
                cmd = [servant_client, 'localhost:{}'.format(provider.apphost_port), context_path]
                logging.info('Running "{}"'.format(cmd))
                try:
                    with open(os.path.join(RESPONSES_FOLDER, context_file), 'wb') as out_file:
                        run_process(cmd, log_prefix='servant_client', outputs_to_one_file=False, stdout=out_file, stderr=err)
                    requests_cnt += 1
                    if requests_cnt >= requests_limit:
                        break
                except Exception as ex:
                    logging.error('Request {} failed: {}'.format(context_file, str(ex)))


__Task__ = EntitySearchApphostResponses
