# -*- coding: utf-8 -*-

import logging
import requests
import time
import urllib2

from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sandboxsdk.parameters import SandboxBoolParameter, SandboxIntegerParameter, SandboxStringParameter
from sandbox.sandboxsdk.ssh import Key
from sandbox.sandboxsdk.svn import Svn

from sandbox.projects import resource_types
from sandbox.projects.common import utils
import sandbox.projects.common.constants as consts
from sandbox.projects.common.search import bugbanner

from sandbox.projects.EntitySearchConvertDb import EntitySearchConvertDb, ConverterResource as EntitySearchConvertDbConverterResource, Table as EntitySearchConvertDbTable
from sandbox.projects.EntitySearchUpdateCmakeResources import EntitySearchUpdateCmakeResources, ConvertedDbTaskParam as UpdateCmakelistsConvertedDbTaskParam,\
    ArcadiaPathParam as UpdateCmakelistsArcadiaPathParam

from sandbox.projects.EntitySearchBinariesBuild import EntitySearchBinariesBuild
from sandbox.projects.EntitySearchConfigBuild import EntitySearchConfigBuild
from sandbox.projects.EntitySearchDataBuild import EntitySearchDataBuild


ENTITYSEARCH_WEB_SVN_URL = 'https://arc.yandex-team.ru/wsvn/arc/branches/entitysearch'
ENTITYSEARCH_BRANCHES_URL = 'svn+ssh://arcadia.yandex.ru/arc/branches/entitysearch'
TRUNK_URL = 'svn+ssh://arcadia.yandex.ru/arc/trunk'
DATABASE_VERSION_TO_PATH_FORMAT_STRING = 'home/dict/ontodb/daily/{}/all_cards_final'


def get_branch_url(branch_number):
    return '{}/stable-{:03d}'.format(ENTITYSEARCH_BRANCHES_URL, branch_number)


def get_arcadia_branch_url(branch_number):
    return get_branch_url(branch_number) + '/arcadia'


def get_web_svn_branch_url(branch_number):
    return '{}/stable-{:03d}'.format(ENTITYSEARCH_WEB_SVN_URL, branch_number)


class BranchNumberParam(SandboxIntegerParameter):
    name = 'branch_number'
    description = 'Optional branch number. 0 - auto'
    default_value = 0


class CreateBranchParam(SandboxBoolParameter):
    name = 'create_branch'
    description = 'Create new branch'
    default_value = True


class TicketParam(SandboxStringParameter):
    name = 'ticket'
    description = 'Startrek ticket (OBJECTS-XXXX)'


class CreateTicketParam(SandboxBoolParameter):
    name = 'create_ticket'
    description = 'Create ticket'
    default_value = True
    sub_fields = {
        'false': [TicketParam.name]
    }


class EntitySearchDatabaseMapreduceTablePathParam(SandboxStringParameter):
    name = 'entitysearch_database_mr_table_path'
    description = 'EntitySearch database table path on mapreduce or db version (for example, "0.335.2017-01-28.00h15m"). Leave empty for current production'


class ReconvertDatabaseParam(SandboxBoolParameter):
    name = 'reconvert_database'
    description = 'Convert database with new convertor'
    default_value = True
    sub_fields = {
        'true': [EntitySearchDatabaseMapreduceTablePathParam.name]
    }


class BuildConfigParam(SandboxBoolParameter):
    name = 'build_config'
    description = 'Build config'
    default_value = True


class SendSmsNotificationParam(SandboxStringParameter):
    name = 'send_sms_notification'
    description = 'Send sms when done to these people (comma separated). Default value is task author'


class ReleaseToTesting(SandboxBoolParameter):
    name = 'do_release'
    description = 'Release to testing'
    default_value = False


class EntitySearchBraveNewWorld(bugbanner.BugBannerTask):
    type = 'ENTITYSEARCH_BRAVE_NEW_WORLD'

    input_parameters = (BranchNumberParam, CreateBranchParam,
                        CreateTicketParam, TicketParam,
                        ReconvertDatabaseParam, EntitySearchDatabaseMapreduceTablePathParam,
                        BuildConfigParam, SendSmsNotificationParam, ReleaseToTesting
                        )

    environment = [PipEnvironment("startrek_client", use_wheel=True)]

    def initCtx(self):
        self.ctx['fail_on_any_error'] = True

        if not self.ctx.get(SendSmsNotificationParam.name):
            self.ctx[SendSmsNotificationParam.name] = self.author

        if not self.ctx.get(EntitySearchDatabaseMapreduceTablePathParam.name):
            try:
                self.ctx[EntitySearchDatabaseMapreduceTablePathParam.name] = self.get_production_database_version()
            except:
                pass

    def get_task_url(self, task_id):
        task = channel.sandbox.get_task(task_id)
        return task.url if task else ''

    def get_task_html_link(self, task_id, title):
        url = self.get_task_url(task_id)
        if url:
            return '<a href="{}">{}</a>'.format(url, title)
        else:
            return title

    def get_next_branch_number(self):
        existing_branches = Svn.list(ENTITYSEARCH_BRANCHES_URL)
        max_num = 0
        # List all branches in directory, than parse their numbers.
        # Our branches have form: <stable/prestable>-DDD
        for branch in existing_branches.split():
            if branch.endswith('/'):
                branch = branch[:-1]
            if branch.find('stable') == -1:
                continue
            if branch[-4] == '-' and branch[-3:].isdigit():
                num = int(branch[-3:])
                if num > max_num:
                    max_num = num
        if max_num:
            return max_num + 1

    def create_branch(self, branch_num):
        if self.ctx.get('branch_created'):
            return

        with Key(self, 'robot-ontodb', 'robot-ontodb-ssh-key'):
            new_branch = get_branch_url(branch_num)
            # Check existence of url
            if Svn.info(new_branch):
                return

            Svn.copy(
                src=TRUNK_URL,
                dst=new_branch,
                message='Created entitysearch-stable-{:03d}.\nTask: {}'.format(branch_num, self.get_task_url(self.id))
            )
            self.set_info('Created stable branch: <a href="{}">{}</a>'.format(get_web_svn_branch_url(branch_num), new_branch), do_escape=False)
            self.ctx['branch_created'] = True

    def get_startrek_description(self):
        brave_new_world_task = 'Brave new world:\n(({} {}))'.format(self.get_task_url(self.id), EntitySearchBraveNewWorld.type)
        ret = brave_new_world_task

        executables_build_task = '\nBinaries:\n(({} {}))'.format(self.get_task_url(self.ctx['executables_build_task']), EntitySearchBinariesBuild.type)
        ret += executables_build_task

        if self.ctx.get('data_build_task'):
            data_build_task = '\nData:\n(({} {}))'.format(self.get_task_url(self.ctx['data_build_task']), EntitySearchDataBuild.type)
            ret += data_build_task

        if self.ctx.get('config_build_task'):
            config_build_task = '\nConfig:\n(({} {}))'.format(self.get_task_url(self.ctx['config_build_task']), EntitySearchConfigBuild.type)
            ret += config_build_task

        branch_url = '\nBranch:\n(({} {}))'.format(get_web_svn_branch_url(self.ctx[BranchNumberParam.name]), get_branch_url(self.ctx[BranchNumberParam.name]))
        ret += branch_url

        return ret

    def create_ticket(self):
        if self.ctx.get('ticket_processed'):
            return

        import startrek_client
        token = self.get_vault_data('robot-ontodb', 'robot-ontodb-st-token')
        st = startrek_client.Startrek(token=token, useragent='sandbox-task')
        branch_number = self.ctx[BranchNumberParam.name]
        ticket = st.issues.create(
            queue='OBJECTS',
            summary='Релиз EntitySearch-{}'.format(branch_number),
            description=self.get_startrek_description(),
            type='task',
            components="Релизы бекенда",
            tags=['ES-{}'.format(branch_number)],
            followers=[
                'radix',
                'ilyagr', 'fminlos',
                'ekaterinas', 'denisbykov', 'karavashkin',
                'daflight', 'phil-grab',
                'konovodov', 'nesko'
            ]
        )
        self.set_info('Created <a href="https://st.yandex-team.ru/{issue_id}">{issue_id}</a>'.format(issue_id=ticket.key), do_escape=False)
        self.ctx[TicketParam.name] = ticket.key
        self.ctx['ticket_processed'] = True

    def comment_to_ticket(self):
        if self.ctx.get('ticket_processed'):
            return

        if not self.ctx.get(TicketParam.name):
            return

        import startrek_client
        token = self.get_vault_data('robot-ontodb', 'robot-ontodb-st-token')
        st_client = startrek_client.Startrek(token=token, useragent='sandbox-task')
        issue = st_client.issues[self.ctx[TicketParam.name]]
        issue.comments.create(text=self.get_startrek_description())

        self.ctx['ticket_processed'] = True

    def get_production_database_version(self):
        attempts = 5
        while attempts:
            attempts = attempts - 1
            try:
                ver = urllib2.urlopen('http://hahn.yt.yandex.net/api/v3/get?path=//home/dict/ontodb/ver/main/production/all_cards_final/@path').read().strip('"')
                logging.info('Found db version: "{}"'.format(ver))
                if ver.startswith('//'):
                    ver = ver[2:]
                return ver
            except Exception as ex:
                logging.error('Failed to read backend database version info: {}'.format(str(ex)))

            time.sleep(20)
        raise Exception('Failed to define production db version. No attempts left')

    def create_reconvert_database_task(self):
        if self.ctx.get('reconvert_db_task'):
            return

        mr_path = self.ctx.get(EntitySearchDatabaseMapreduceTablePathParam.name)
        if not mr_path:
            mr_path = self.get_production_database_version()
            self.ctx[EntitySearchDatabaseMapreduceTablePathParam.name] = mr_path
        elif mr_path.find('/') == -1:
            # path from version
            mr_path = DATABASE_VERSION_TO_PATH_FORMAT_STRING.format(mr_path)

        convertor_res = channel.sandbox.list_resources(
            resource_type=resource_types.ENTITY_SEARCH_CONVERTER,
            task_id=self.ctx['executables_build_task'],
            limit=None
        )
        if not convertor_res:
            raise SandboxTaskFailureError('Failed to get convertor resource')

        input_parameters = {
            EntitySearchConvertDbTable.name: mr_path,
            EntitySearchConvertDbConverterResource.name: convertor_res[0].id,
        }

        self.ctx['reconvert_db_task'] = self.create_subtask(
            task_type=EntitySearchConvertDb.type,
            description='Convert database task. Version: {}'.format(mr_path),
            input_parameters=input_parameters
        ).id

        self.set_info('Created {}'.format(self.get_task_html_link(self.ctx['reconvert_db_task'], EntitySearchConvertDb.type)), do_escape=False)

    def wait_reconvert_database_task(self):
        utils.check_tasks_to_finish_correctly(task_ids_list=[self.ctx['reconvert_db_task']])

    def run_reconvert_database_task(self):
        self.create_reconvert_database_task()
        self.wait_reconvert_database_task()

    def create_update_cmakelists_task(self):
        if self.ctx.get('update_cmakelists_task'):
            return

        input_parameters = {
            UpdateCmakelistsArcadiaPathParam.name: get_arcadia_branch_url(self.ctx[BranchNumberParam.name]),
            UpdateCmakelistsConvertedDbTaskParam.name: self.ctx['reconvert_db_task'],
        }

        self.ctx['update_cmakelists_task'] = self.create_subtask(
            task_type=EntitySearchUpdateCmakeResources.type,
            description='Update CMakeLists.txt for EntitySearch-{}'.format(self.ctx[BranchNumberParam.name]),
            input_parameters=input_parameters
        ).id

    def wait_update_cmakelists_task(self):
        utils.check_tasks_to_finish_correctly(task_ids_list=[self.ctx['update_cmakelists_task']])

    def run_update_cmakelists_task(self):
        self.create_update_cmakelists_task()
        self.wait_update_cmakelists_task()

    def create_executables_build_task(self):
        if self.ctx.get('executables_build_task'):
            return

        input_parameters = {
            consts.ARCADIA_URL_KEY: get_arcadia_branch_url(self.ctx[BranchNumberParam.name]),
            consts.CHECKOUT: True,
        }
        self.ctx['executables_build_task'] = self.create_subtask(
            task_type=EntitySearchBinariesBuild.type,
            description='EntitySearch-{:03d} executables'.format(self.ctx[BranchNumberParam.name]),
            arch='linux',
            input_parameters=input_parameters,
        ).id

        self.set_info('Created {}'.format(self.get_task_html_link(self.ctx['executables_build_task'], EntitySearchBinariesBuild.type)), do_escape=False)

    def wait_executables_build_task(self):
        utils.check_tasks_to_finish_correctly(task_ids_list=[self.ctx['executables_build_task']])

    def run_executables_build_task(self):
        self.create_executables_build_task()
        self.wait_executables_build_task()

    def create_config_build_task(self):
        if self.ctx.get('config_build_task'):
            return

        input_parameters = {
            consts.ARCADIA_URL_KEY: get_arcadia_branch_url(self.ctx[BranchNumberParam.name]),
            'kill_timeout': 3600 * 6,
        }
        self.ctx['config_build_task'] = self.create_subtask(
            task_type=EntitySearchConfigBuild.type,
            description='EntitySearch config',
            input_parameters=input_parameters,
        ).id

        self.set_info('Created {}'.format(self.get_task_html_link(self.ctx['config_build_task'], EntitySearchConfigBuild.type)), do_escape=False)

    def wait_config_build_task(self):
        utils.check_tasks_to_finish_correctly(task_ids_list=[self.ctx['config_build_task']])

    def run_config_build_task(self):
        self.create_config_build_task()
        self.wait_config_build_task()

    def create_data_build_task(self):
        if self.ctx.get('data_build_task'):
            return

        input_parameters = {
            consts.ARCADIA_URL_KEY: get_arcadia_branch_url(self.ctx[BranchNumberParam.name]),
            consts.CHECKOUT: True,
        }
        self.ctx['data_build_task'] = self.create_subtask(
            task_type=EntitySearchDataBuild.type,
            description='EntitySearch-{:03d} data'.format(self.ctx[BranchNumberParam.name]),
            arch='linux',
            input_parameters=input_parameters
        ).id

        self.set_info('Created {}'.format(self.get_task_html_link(self.ctx['data_build_task'], EntitySearchDataBuild.type)), do_escape=False)

    def wait_data_build_task(self):
        utils.check_tasks_to_finish_correctly(task_ids_list=[self.ctx['data_build_task']])

    def run_data_build_task(self):
        self.create_data_build_task()
        self.wait_data_build_task()

    def run_shooter_tasks(self):
        pass  # not implemented

    def release_task_to_testing(self, task_id, task_type):
        if task_id:
            self.create_release(
                task_id=task_id,
                status='testing'
            )

            self.set_info('{} released to testing'.format(self.get_task_html_link(task_id, task_type)), do_escape=False)

    def release_to_testing(self):
        self.release_task_to_testing(self.ctx.get('executables_build_task', 0), EntitySearchBinariesBuild.type)
        self.release_task_to_testing(self.ctx.get('data_build_task', 0), EntitySearchDataBuild.type)
        self.release_task_to_testing(self.ctx.get('config_build_task', 0), EntitySearchConfigBuild.type)

    def get_sms_text(self, succeeded):
        if succeeded:
            return 'Новый бекенд готов к приемке. {}. Task id: {}'.format(self.ctx.get(TicketParam.name), self.id)
        else:
            return 'Brave new world завершился с ошибкой. Task id: {}'.format(self.id)

    def send_sms_notification(self, succeeded=True):
        users = self.ctx.get(SendSmsNotificationParam.name).replace(' ', '')
        if users:
            URL = "https://golem.yandex-team.ru/api/sms/send.sbml?resps={users}&msg={text}"
            url = URL.format(users=users, text=urllib2.quote(self.get_sms_text(succeeded)))
            requests.get(url)

    def on_execute(self):
        self.add_bugbanner(bugbanner.Banners.EntitySearch)

        if not self.ctx.get(BranchNumberParam.name):
            self.ctx[BranchNumberParam.name] = self.get_next_branch_number()

        if self.ctx.get(CreateBranchParam.name):
            self.create_branch(self.ctx[BranchNumberParam.name])

        self.run_executables_build_task()

        if self.ctx.get(ReconvertDatabaseParam.name):
            self.run_reconvert_database_task()
            self.run_update_cmakelists_task()
            self.run_data_build_task()

        if self.ctx.get(BuildConfigParam.name):
            self.run_config_build_task()

        self.run_shooter_tasks()

        if self.ctx.get(CreateTicketParam.name):
            self.create_ticket()
        else:
            self.comment_to_ticket()

        if self.ctx.get(ReleaseToTesting.name):
            self.release_to_testing()

    def on_success(self):
        bugbanner.BugBannerTask.on_success(self)
        if self.ctx.get(SendSmsNotificationParam.name):
            self.send_sms_notification()

    def on_failure(self):
        bugbanner.BugBannerTask.on_failure(self)
        if self.ctx.get(SendSmsNotificationParam.name):
            self.send_sms_notification(False)


__Task__ = EntitySearchBraveNewWorld
