# -*- coding: utf-8 -*-

import itertools
import os

from sandbox.projects import resource_types

from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk.errors import SandboxTaskFailureError

from sandbox.projects.common.search import bugbanner
from sandbox.projects.common.differ import printers, json_differ


DISK_SPACE_REQUIREMENTS = 10 * 1024  # 10 GB


def get_responses_param(i):
    class Responses(parameters.ResourceSelector):
        name = "responses_" + str(i)
        description = "Apphost context responses " + str(i)
        resource_type = resource_types.ENTITY_SEARCH_APPHOST_RESPONSES
        required = True
    return Responses


class EntitySearchCompareApphostResponses(bugbanner.BugBannerTask):
    """
        Сравнение ответов двух бекендов entitysearch
    """

    type = 'ENTITYSEARCH_COMPARE_APPHOST_RESPONSES'

    execution_space = DISK_SPACE_REQUIREMENTS

    input_parameters = [get_responses_param(1), get_responses_param(2)]

    def load_responses(self, res_id):
        res_path = self.sync_resource(res_id)
        responses = []
        for response in sorted(os.listdir(res_path)):
            with open(os.path.join(res_path, response)) as f:
                responses.append(f.read())
        return responses

    def on_execute(self):
        self.add_bugbanner(bugbanner.Banners.EntitySearch, component="differ")

        responses1 = self.load_responses(self.ctx.get('responses_1'))
        responses2 = self.load_responses(self.ctx.get('responses_2'))
        if len(responses1) != len(responses2):
            raise SandboxTaskFailureError('Responses are not compatible')

        output_dir = 'diff'
        os.mkdir(output_dir)
        self.create_resource('diff', output_dir, resource_types.ENTITY_SEARCH_DIFF)

        printer = printers.PrinterToHtml(output_dir,
                                         write_compact_diff=True,
                                         pair_head_template="response {obj_index}",
                                         obj_head_template="object")
        differ = json_differ.JsonDiffer(printer)
        cmp_data = itertools.izip(
            responses1,
            responses2,
            itertools.repeat('obj_id'),  # TODO: retrieve object id and name + query if exists
        )

        differ.compare_pairs(cmp_data)

        self.ctx["diff_count"] = differ.get_diff_count()
        self.ctx["compare_result"] = differ.get_diff_count() == 0


__Task__ = EntitySearchCompareApphostResponses
