# -*- coding: utf-8 -*-

import logging
import os
import shutil

from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.sandboxsdk import parameters

from sandbox.projects import resource_types
from sandbox.projects.common.nanny import nanny
from sandbox.projects.common.build.parameters import ArcadiaUrl, ArcadiaPatch
import sandbox.projects.common.constants as consts
from sandbox.sandboxsdk import errors


MIN_DISK_SPACE = 50 * 1024  # 50 GB for Arcadia cache


class DirectFileCheckoutOnNoPatch(parameters.SandboxBoolParameter):
    name = 'direct_file_checkout_on_no_patch'
    description = 'Do direct file checkout if no patch provided'
    default_value = True


class EntitySearchConfigBuild(nanny.ReleaseToNannyTask, SandboxTask):
    config_arcadia_relative_path = os.path.join('search', 'wizard', 'entitysearch', 'data', 'production.cfg')

    type = 'ENTITYSEARCH_CONFIG_BUILD'

    input_parameters = [ArcadiaUrl, ArcadiaPatch, DirectFileCheckoutOnNoPatch]

    execution_space = MIN_DISK_SPACE

    def on_release(self, additional_parameters):
        nanny.ReleaseToNannyTask.on_release(self, additional_parameters)
        SandboxTask.on_release(self, additional_parameters)

    def get_arcadia_src_dir(self):
        arcadia_src_dir = Arcadia.get_arcadia_src_dir(
            self.ctx[consts.ARCADIA_URL_KEY])
        if not arcadia_src_dir:
            raise errors.SandboxTaskFailureError(
                'Cannot get repo for url {0}'.format(self.ctx[consts.ARCADIA_URL_KEY])
            )
        return arcadia_src_dir

    def on_execute(self):
        patch = self.ctx.get(consts.ARCADIA_PATCH_KEY)

        if patch or not self.ctx[DirectFileCheckoutOnNoPatch.name]:
            config_path = self.get_config_with_full_checkout_and_patch(patch)
        else:
            config_path = self.get_config_with_direct_cat()

        config_dest_file_name = 'production.cfg'
        shutil.copyfile(config_path, config_dest_file_name)
        self.create_resource('EntitySearch Config', config_dest_file_name, resource_types.ENTITY_SEARCH_CONFIG)

    def get_config_with_full_checkout_and_patch(self, patch_value):
        arcadia = self.get_arcadia_src_dir()

        if patch_value:
            Arcadia.apply_patch(arcadia, patch_value, self.abs_path())

        return os.path.join(arcadia, EntitySearchConfigBuild.config_arcadia_relative_path)

    def get_config_with_direct_cat(self):
        result_file = 'cat_production.cfg'

        input_arcadia_url = self.ctx[consts.ARCADIA_URL_KEY]

        parsed_url = Arcadia.parse_url(input_arcadia_url)
        config_parsed_url = parsed_url._replace(subpath=EntitySearchConfigBuild.config_arcadia_relative_path)

        config_url_to_cat = os.path.join(
            Arcadia.ARCADIA_SCHEME + ':',
            config_parsed_url.path,
            config_parsed_url.subpath,
        )

        logging.debug('Checkouting file directly from {url}, rev. r{rev}'.format(url=config_url_to_cat, rev=config_parsed_url.revision))

        with open(result_file, 'w') as h:
            h.writelines(Arcadia.cat(config_url_to_cat, revision=config_parsed_url.revision))

        return result_file


__Task__ = EntitySearchConfigBuild
