# -*- coding: utf-8 -*-

import os
import shutil

from sandbox.sandboxsdk import process
from sandbox.sandboxsdk.parameters import SandboxStringParameter, ResourceSelector, SandboxSelectParameter, SandboxBoolParameter

from sandbox.projects.common.wizard.utils import tar
from sandbox.projects.common.wizard.current_production import get_current_production_task_id, get_resource_id_from_task
from sandbox.projects.common.search import bugbanner
from sandbox.projects.EntitySearch import resource_types

import sandbox.common.types.client as ctc

ES_TRIE_TYPE = 'ENTITY_SEARCH_TRIE'

GROUP_NAME = 'Convert db params'

MIN_RAM_FOR_CONVERTOR = 100 * 1024  # 100 GB
MIN_DISK_SPACE_FOR_CONVERTOR = 128 * 1024  # 128 GB

DEFAULT_CODEC = 'default'
NO_CODEC = 'none'

CODEC_OPTS = [DEFAULT_CODEC, NO_CODEC]
LOG_LEVEL_OPTS = ['ERROR', 'WARNING', 'INFO', 'DEBUG']


class ServerName(SandboxStringParameter):
    name = 'server'
    description = 'Server address'
    default_value = 'hahn.yt.yandex.net'
    group = GROUP_NAME


class Table(SandboxStringParameter):
    name = 'table'
    description = 'Table name or version'
    group = GROUP_NAME


class ConverterResource(ResourceSelector):
    name = 'converter_res'
    description = 'Converter'
    required = False
    resource_type = resource_types.ENTITY_SEARCH_CONVERTER
    group = GROUP_NAME


class ResultPrefix(SandboxStringParameter):
    name = 'result_prefix'
    description = 'Result prefix'
    default_value = 'main'
    group = GROUP_NAME


class CodecType(SandboxSelectParameter):
    name = 'codec'
    description = 'codec to use in convertor'
    choices = tuple((opt, opt) for opt in CODEC_OPTS)
    default_value = DEFAULT_CODEC
    group = GROUP_NAME


class SplitCard(SandboxBoolParameter):
    name = 'split_card'
    description = "'split-card' flag from converter (only for main base)"
    default_value = False
    group = GROUP_NAME


class UseProductionYtQoutaForTmp(SandboxBoolParameter):
    name = 'use_production_yt_quota_for_tmp'
    description = "overrides '--tmp-table-prefix' to production quota YT directory"
    default_value = True
    group = GROUP_NAME


class YtLogLevel(SandboxSelectParameter):
    name = 'yt_log_level'
    description = 'value to set in YT_LOG_LEVEL env'
    choices = tuple((opt, opt) for opt in LOG_LEVEL_OPTS)
    default_value = 'DEBUG'
    group = GROUP_NAME


class MakeArtifactsArchive(SandboxBoolParameter):
    name = 'make_artifacts_archive'
    description = 'flag for registering main conversion artifacts as one resource'
    default_value = False
    group = GROUP_NAME


# TODO: change for better way finding production resources
PRODUCTION_ENTITYSEARCH_NANNY_SERVICE = 'sas-production-entitysearch-yp'


class EntitySearchConvertDb(bugbanner.BugBannerTask):
    type = 'ENTITYSEARCH_CONVERT_DB'
    client_tags = ctc.Tag.Group.LINUX

    input_parameters = [
        ServerName, Table, ConverterResource, ResultPrefix, CodecType, YtLogLevel, SplitCard, UseProductionYtQoutaForTmp, MakeArtifactsArchive
    ]

    required_ram = MIN_RAM_FOR_CONVERTOR
    execution_space = MIN_DISK_SPACE_FOR_CONVERTOR

    def init_env(self):
        os.environ['MR_RUNTIME'] = 'YT'
        os.environ['YT_LOG_LEVEL'] = self.ctx[YtLogLevel.name]
        os.environ['YT_CLIENT_TIMEOUT'] = '300'

        path = os.path.expanduser('~/.yt/')
        if not os.path.exists(path):
            os.makedirs(path)

        with open(os.path.expanduser('~/.yt/token'), 'w') as token:
            token.write(self.get_vault_data('robot-ontodb', 'robot-ontodb-yt-token'))

    def get_converter(self):
        if not self.ctx[ConverterResource.name]:
            task_id = get_current_production_task_id(
                PRODUCTION_ENTITYSEARCH_NANNY_SERVICE,
                resource_types.ENTITY_SEARCH_EXECUTABLE,
                self.get_vault_data('robot-ontodb', 'nanny-oauth-token')
            )
            converter_id = get_resource_id_from_task(task_id, resource_types.ENTITY_SEARCH_CONVERTER)
            self.ctx[ConverterResource.name] = converter_id
        return self.sync_resource(self.ctx[ConverterResource.name])

    def convert(self):
        table = self.ctx[Table.name]
        table_name = 'home/dict/ontodb/ver/main/production/all_cards_final'

        if table:
            table_name = 'home/dict/ontodb/ver/main/0.{}/all_cards_final'.format(table) if table.isdigit() else table

        cmd = '''{cnv_bin} -s {src_table} --server {yt_srv} --card-filter=shown-only {tmp_table_prefix}
                --external . -D . -n {pref}{codec}{split_card}'''.format(
            cnv_bin=self.get_converter(),
            src_table=table_name,
            yt_srv=self.ctx[ServerName.name],
            tmp_table_prefix='--tmp-table-prefix home/dict/ontodb/entitysearch_convertor' if self.ctx[UseProductionYtQoutaForTmp.name] else '',
            pref=self.ctx[ResultPrefix.name],
            codec='' if self.ctx[CodecType.name] == CodecType.default_value else ' --codec ' + self.ctx[CodecType.name],
            split_card=' --split-card' if self.ctx[SplitCard.name] is True else '',
        )
        process.run_process(cmd.split(), log_prefix='convertor', outputs_to_one_file=True)

    def move_files_register_archive_resource(self):
        name = self.ctx[ResultPrefix.name]

        archive_dir = './' + name + '_db'
        os.mkdir(archive_dir)

        for file_extention in ('trie', 'gzt.bin', 'remap.trie'):
            conversion_artifact = name + '.' + file_extention
            shutil.move(conversion_artifact, archive_dir)

        archive_external_dir = os.path.join(archive_dir, 'external')
        os.mkdir(archive_external_dir)
        for external_trie in ('yamusic_ids.trie', 'yasport_ids.trie', 'google_play_ids.trie', 'itunes_ids.trie'):
            shutil.move(external_trie, archive_external_dir)

        self.create_resource(
            description='archive for {}'.format(self.ctx[Table.name]),
            resource_path=archive_dir,
            resource_type=resource_types.ENTITY_SEARCH_CONVERT_DB_ARCHIVE,
        )

    def register_tar_resource(self, descr, path, resource_type, attributes=None):
        if os.path.exists(path):
            self.create_resource(descr, tar(path), resource_type, attributes=attributes)

    def register_resources(self):
        name = self.ctx[ResultPrefix.name]

        gzt_bin_file_name = '{}.gzt.bin'.format(name)
        self.register_tar_resource('main gazeteer', gzt_bin_file_name, 'ENTITY_SEARCH_GAZETEER', attributes={'db_type': gzt_bin_file_name, 'ttl': 'inf'})

        trie_file_name = '{}.trie'.format(name)
        self.register_tar_resource('main trie', trie_file_name, resource_types.ENTITY_SEARCH_TRIE, attributes={'db_type': name, 'ttl': 'inf'})

        remap_trie_file_name = '{}.remap.trie'.format(name)
        self.register_tar_resource('remap trie', remap_trie_file_name, resource_types.ENTITY_SEARCH_TRIE, attributes={'db_type': '{}_remap_trie'.format(name), 'ttl': 'inf'})

        wizard_trie_dbtype = '{}.wizard'.format(name)
        wizard_trie_file_name = '{}.wizard.trie'.format(name)
        self.register_tar_resource('wizard trie', wizard_trie_file_name, resource_types.ENTITY_SEARCH_TRIE, attributes={'db_type': wizard_trie_dbtype, 'ttl': 'inf'})

        self.register_tar_resource('yamusic ids trie', 'yamusic_ids.trie', resource_types.ENTITY_SEARCH_TRIE, attributes={'db_type': 'yamusic_ids', 'ttl': 'inf'})
        self.register_tar_resource('yasport ids trie', 'yasport_ids.trie', resource_types.ENTITY_SEARCH_TRIE, attributes={'db_type': 'yasport_ids', 'ttl': 'inf'})
        self.register_tar_resource('google play ids trie', 'google_play_ids.trie', resource_types.ENTITY_SEARCH_TRIE, attributes={'db_type': 'google_play_ids', 'ttl': 'inf'})
        self.register_tar_resource('itunes ids trie', 'itunes_ids.trie', resource_types.ENTITY_SEARCH_TRIE, attributes={'db_type': 'itunes_ids', 'ttl': 'inf'})

        if self.ctx[MakeArtifactsArchive.name]:
            self.move_files_register_archive_resource()

    def on_execute(self):
        self.add_bugbanner(bugbanner.Banners.EntitySearch, component="entitysearch_convertor")

        self.init_env()
        self.convert()
        self.register_resources()


__Task__ = EntitySearchConvertDb
