# -*- coding: utf-8 -*-

import logging

from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sandboxsdk.process import run_process

from sandbox.projects import resource_types
import sandbox.common.types.client as ctc
from sandbox.projects.common.wizard.providers import EntitySearchProvider
from sandbox.projects.common.wizard.providers import EntitySearchBinary
from sandbox.projects.common.wizard.providers import EntitySearchConfig
from sandbox.projects.common.wizard.providers import EntitySearchData
from sandbox.projects.common.wizard.providers import Fresh
from sandbox.projects.common.wizard.providers import NerData

from sandbox.projects.EntitySearch.common.current_production import get_current_production_resource_id_from_service

from sandbox.projects.common import utils
from sandbox.projects.common.search import performance as search_performance
from sandbox.projects.common.search import bugbanner
from sandbox.projects.common.dolbilka import DolbilkaPlanner


class Reqs(parameters.ResourceSelector):
    name = "reqs"
    description = "Query list textfile (plain text, empty to autodetect)"
    resource_type = resource_types.PLAIN_TEXT_QUERIES


class SaveDaemonVersion(parameters.SandboxBoolParameter):
    name = "save_daemon_version"
    description = "Saves <daemon_host>:<daemon_port>/v output to context"
    default = False


def magic_avg(vals):
    return (sum(vals) - min(vals)) / (len(vals) - 1)


class EntitySearchSharpShooter(search_performance.OldShootingTask, bugbanner.BugBannerTask):
    type = 'ENTITYSEARCH_SHARP_SHOOTER'

    required_ram = 140 * 1024
    execution_space = EntitySearchProvider.DISK_SPACE_FOR_DATA
    client_tags = ctc.Tag.LINUX_PRECISE & ctc.Tag.INTEL_E5_2650

    production_service = 'sas-production-entitysearch-yp'

    # small but dirty hack
    es_params = EntitySearchProvider.input_parameters
    for param in es_params:
        param.required = False

    input_parameters = [
        Reqs,
        SaveDaemonVersion,
    ] + es_params + list(search_performance.OldShootingTask.shoot_input_parameters)

    current_daemon = None

    # stats from tank.DolbiloPlugin
    new_stats_types = (
        ("latency_0.5", "Latency, q50 (usec)", "{:0.2f}"),
        ("latency_0.95", "Latency, q95 (usec)", "{:0.2f}"),
        ("latency_0.99", "Latency, q99 (usec)", "{:0.2f}"),
        ("resp_size_quantile_0.5", "Response size, q50", "{:0.0f}"),
        ("resp_size_quantile_0.95", "Response size, q95", "{:0.0f}"),
        ("max_size", "Max size", "{:0.0f}"),
    )

    def make_plan(self):
        plan_path = self.abs_path('plan')
        path_to_planner = DolbilkaPlanner.get_planner_path()
        requests = self.sync_resource(self.ctx['reqs'])
        command = ' '.join([path_to_planner, '-l', requests, '-o', plan_path, '-t plain',
                            '-h', self.current_daemon.host, '-p', str(self.current_daemon.port)])
        logging.info(command)
        run_process(command, log_prefix='planner')
        resource = self.create_resource('wizard plan', plan_path, resource_types.REMOTE_WIZARD_PLAN)
        self.mark_resource_ready(resource)
        self.ctx['dolbilo_plan_resource_id'] = resource.id

    def _start_searcher(self, callbacks=None):
        if not self.current_daemon.alive():
            raise SandboxTaskFailureError('ES is dead. R.I.P.')
        return [self.current_daemon.process]

    def _get_searcher_port(self):
        return self.current_daemon.port

    def _get_unspecified_resources_from_production(self):
        required_resources = [EntitySearchBinary, EntitySearchConfig, EntitySearchData, Fresh, NerData]

        nanny_token = None
        for resource in required_resources:
            res_name = resource.name
            res_type = resource.resource_type
            if self.ctx[res_name] is None:
                if nanny_token is None:
                    nanny_token = self.get_vault_data('robot-ontodb', 'nanny-oauth-token')

                self.ctx[res_name] = get_current_production_resource_id_from_service(
                    self.production_service,
                    res_type,
                    nanny_token
                )
                logging.info('used resource with id {} as {}'.format(self.ctx[res_name], res_name))

    def on_execute(self):
        self.add_bugbanner(bugbanner.Banners.EntitySearch)

        if not self.ctx['reqs']:
            self.ctx['reqs'] = utils.get_and_check_last_resource_with_attribute(
                'PLAIN_TEXT_QUERIES', 'entitysearch_requests', '1'
            ).id

        self._get_unspecified_resources_from_production()

        nanny_token = self.get_vault_data('robot-ontodb', 'nanny-oauth-token')
        with EntitySearchProvider.from_task_context(self.ctx, nanny_token=nanny_token) as provider:
            self.current_daemon = provider
            self.ctx["entitysearch_load_time"] = provider.get_load_time()

            if self.ctx["save_daemon_version"]:
                self.ctx["daemon_version"] = provider.get_version()

            self.make_plan()

            self._init_virtualenv()
            self._old_shoot(provider, self.ctx['dolbilo_plan_resource_id'])


__Task__ = EntitySearchSharpShooter
