# -*- coding: utf-8 -*-

import copy
import logging
import datetime

import sandbox.common.types.client as ctc

from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk.process import run_process, get_process_info
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sandboxsdk.channel import channel

from sandbox.projects import resource_types
from sandbox.projects.common.BaseTestTask import BaseDolbiloTask
from sandbox.projects.common.dolbilka import DolbilkaPlanner
from sandbox.projects.common.wizard.providers import EntitySearchProvider
from sandbox.projects.common.wizard.wizard_mailer import WizardMailer
from sandbox.projects.common import apihelpers
from sandbox.projects.EntitySearchShooter.object_quality_checker import quality_cards_count

QUALITY_CHECKING_GROUP_NAME = "Quality checking"


class Reqs(parameters.ResourceSelector):
    name = "reqs"
    description = "Query list textfile (plain text, empty to autodetect)"
    resource_type = resource_types.PLAIN_TEXT_QUERIES
    attrs = {'entitysearch_requests': 1}


class CheckQuality(parameters.SandboxBoolParameter):
    name = "check_quality"
    description = "Check quality"
    default_value = False
    group = QUALITY_CHECKING_GROUP_NAME


class ObjectIdsList(parameters.ResourceSelector):
    name = "object_ids_list"
    description = "Object ids list"
    resource_type = resource_types.ENTITY_SEARCH_OBJECT_IDS_LIST
    group = QUALITY_CHECKING_GROUP_NAME


class QualityThreshold(parameters.SandboxIntegerParameter):
    name = "threshold"
    description = "Good cards percent threshold"
    default_value = 75
    group = QUALITY_CHECKING_GROUP_NAME


class Notify(parameters.SandboxBoolParameter):
    name = "notify"
    description = "send to entity-search-tests@"


class EntitySearchShooter(BaseDolbiloTask):
    type = 'ENTITYSEARCH_SHOOTER'

    required_ram = EntitySearchProvider.MIN_RAM
    execution_space = EntitySearchProvider.DISK_SPACE_FOR_DATA

    input_parameters = [
        Reqs,
        CheckQuality,
        ObjectIdsList,
        QualityThreshold,
    ] + EntitySearchProvider.input_parameters + list(BaseDolbiloTask.input_parameters)

    client_tags = ctc.Tag.LINUX_PRECISE

    current_daemon = None

    def initCtx(self):
        self.ctx.update(
            {
                'requests_limit': 200000,
                'total_sessions': 5,
                'executor_mode': 'finger',
                'fuckup_mode_max_simultaneous_requests': 15,
                'plan_mode_delay_between_requests_multipliers': 1.0,
            }
        )

    @property
    def footer(self):
        return 'rps: {}<br/>mem: {}<br/>'.format(
            self.ctx.get('rps'), self.ctx.get('mem')
        ) if self.is_finished() else ''

    def _start_searcher(self, callbacks=None):
        logging.info('current pid %s' % [self.current_daemon.process.pid])
        if not self.current_daemon.alive():
            raise SandboxTaskFailureError('wizard is dead')
        return [self.current_daemon.process]

    def _stop_searcher(self, subproc_list, session_id=None):
        logging.info('ignoring _stop_searcher')

    def _get_searcher_port(self):
        return self.current_daemon.port

    def make_plan(self):
        plan_path = self.abs_path('plan')
        path_to_planner = DolbilkaPlanner.get_planner_path()
        requests = self.sync_resource(self.ctx['reqs'])
        command = ' '.join([path_to_planner, '-l', requests, '-o', plan_path, '-t plain', "-h yandex.ru -p 80"])
        logging.info(command)
        run_process(command, log_prefix='planner')
        resource = self.create_resource('wizard plan', plan_path, resource_types.REMOTE_WIZARD_PLAN)
        self.mark_resource_ready(resource)
        self.ctx['dolbilo_plan_resource_id'] = resource.id

    def notify(self):
        content = '<a href="https://sandbox.yandex-team.ru/sandbox/tasks/view?task_id={}">task</a> created {}<br>rps: {}<br>mem: {}gb<br>'.format(
            self.id,
            datetime.datetime.fromtimestamp(self.timestamp).strftime('%Y-%m-%d %H:%M'),
            self.ctx['rps'],
            self.ctx['mem']
        )
        WizardMailer.send(
            content=content,
            subj='Entitysearch shooting results',
            recipients=['entity-search-tests@yandex-team.ru'],
        )

    def find_requests(self):
        resources = apihelpers.get_resources_with_attribute(
            resource_type='PLAIN_TEXT_QUERIES',
            attribute_name='entitysearch_requests',
            attribute_value='1',
            limit=50
            )
        if not resources:
            raise SandboxTaskFailureError('No requests are available')

        for res in resources:
            # try to find resource withut concrete client
            if res.is_ready() and 'client' not in res.attributes:
                return res.id
        return resources[0].id

    def find_object_ids(self):
        resource = apihelpers.get_last_resource(resource_types.ENTITY_SEARCH_OBJECT_IDS_LIST)
        if not resource:
            raise SandboxTaskFailureError('No resource ENTITY_SEARCH_OBJECT_IDS_LIST')
        return resource.id

    def on_execute(self):
        if not self.ctx['reqs']:
            self.ctx['reqs'] = self.find_requests()

        if self.ctx[CheckQuality.name] and not self.ctx[ObjectIdsList.name]:
            self.ctx[ObjectIdsList.name] = self.find_object_ids()

        self.ctx['requests_per_sec'] = []
        self.make_plan()

        nanny_token = self.get_vault_data('robot-ontodb', 'nanny-oauth-token')
        with EntitySearchProvider.from_task_context(self.ctx, nanny_token=nanny_token) as provider:
            self.current_daemon = provider
            logging.info('ps info: %s' % get_process_info(self.current_daemon.process.pid, ['rss']))
            params = copy.deepcopy(self.ctx)
            params['requests_per_sec'] = []
            self.run_dolbilo(params, 'entitysearch', [], start_once=True)
            self.ctx['requests_per_sec'] = self.ctx.get('requests_per_sec', []) + params['requests_per_sec']
            self.ctx['mem'] = round(provider.get_memory() / (2.0 ** 30), 3)

            self.ctx['daemon_version'] = provider.get_version()

            if self.ctx[CheckQuality.name]:
                mask = 'http://localhost:{}/get?obj={{}}'.format(provider.port)
                object_ids_file_path = channel.task.sync_resource(self.ctx[ObjectIdsList.name])
                cards_result = quality_cards_count(mask, object_ids_file_path)
                good_cards = cards_result[0]
                total_cards = cards_result[1]
                good_cards_percentage = good_cards * 100 / total_cards
                self.set_info('Got {} good cards of {} ({}%)'.format(good_cards, total_cards, good_cards_percentage))
                self.ctx['good_cards_percentage'] = good_cards_percentage
                if good_cards_percentage < self.ctx[QualityThreshold.name]:
                    raise SandboxTaskFailureError(
                        'Checking quality failure. Got {} good cards of {} ({}%) but {}% is required'.format(
                            good_cards,
                            total_cards,
                            good_cards_percentage,
                            self.ctx[QualityThreshold.name]
                        )
                    )

        avg = lambda vals: ((sum(vals) - min(vals)) / (len(vals) - 1)) if len(vals) > 1 else vals[0]
        self.ctx['rps'] = round(avg(self.ctx['requests_per_sec']))
        self.set_info('rps: {}'.format(self.ctx['rps']))

        if self.ctx.get('notify'):
            self.notify()


__Task__ = EntitySearchShooter
