# -*- coding: utf-8 -*-

import os

from sandbox.sandboxsdk.svn import Arcadia
from sandbox.sandboxsdk.ssh import Key
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sandboxsdk.parameters import SandboxStringParameter, TaskSelector, SandboxBoolParameter

from sandbox.projects.common.wizard.svn_wizard import SvnWizard
from sandbox.projects.common import apihelpers, utils
from sandbox.projects.common.search import bugbanner

GROUP_NAME = 'Update ya.make params'

BRANCH_CREATED_MARK = 'branch_created'


class ArcadiaPathParam(SandboxStringParameter):
    name = 'arcadia_path'
    description = 'Arcadia path'
    default_value = 'svn+ssh://arcadia.yandex.ru/arc/trunk/arcadia'
    group = GROUP_NAME


class CreateBranchParam(SandboxBoolParameter):
    name = 'create_branch'
    description = 'Create branch if it does not exist'
    group = GROUP_NAME


class ConvertedDbTaskParam(TaskSelector):
    name = 'converted_db_task'
    description = 'Converted db task'
    task_type = 'ENTITYSEARCH_CONVERT_DB'
    group = GROUP_NAME


class ConvertedAcceptedMainDeltaTaskParam(TaskSelector):
    name = 'converted_accepted_main_delta_task'
    description = 'Converted accepted main delta task'
    task_type = 'ENTITYSEARCH_CONVERT_DB'
    group = GROUP_NAME


class SourceBranchParam(SandboxStringParameter):
    name = 'src_branch'
    description = 'Source branch'
    default_value = Arcadia.ARCADIA_TRUNK_URL
    group = GROUP_NAME


class EntitySearchUpdateCmakeResources(bugbanner.BugBannerTask):
    type = 'ENTITYSEARCH_UPDATE_CMAKE_RESOURCES'

    input_parameters = (ArcadiaPathParam, CreateBranchParam, ConvertedDbTaskParam, ConvertedAcceptedMainDeltaTaskParam, SourceBranchParam)

    @staticmethod
    def patch_ya_make(ya_make_path, patches):
        new_ya_make_content = []
        with open(ya_make_path) as ya_make:
            for line in ya_make.readlines():
                patched = False
                for name in patches:
                    if ' OUT {})'.format(name) in line:
                        new_ya_make_content.append('FROM_SANDBOX({} OUT {})\n'.format(patches[name], name))
                        patched = True
                        break
                if not patched:
                    new_ya_make_content.append(line)

        with open(ya_make_path, 'w') as mutable_ya_make:
            mutable_ya_make.write(''.join(new_ya_make_content))

    def get_patches(self):
        patches = {}
        resources = apihelpers.list_task_resources(self.ctx[ConvertedDbTaskParam.name])
        for r in resources:
            if r.type == 'ENTITY_SEARCH_GAZETEER':
                patches['main.gzt.bin'] = r.id
            elif r.type == 'ENTITY_SEARCH_TRIE':
                if 'db_type' in r.attributes:
                    patches[r.attributes['db_type'] + '.trie'] = r.id

        if self.ctx[ConvertedAcceptedMainDeltaTaskParam.name]:
            resources = apihelpers.list_task_resources(self.ctx[ConvertedAcceptedMainDeltaTaskParam.name])
            for r in resources:
                if r.type == 'ENTITY_SEARCH_GAZETEER':
                    patches[r.attributes['db_type']] = r.id
                elif r.type == 'ENTITY_SEARCH_TRIE' and r.file_name.startswith('accepted_main_delta.trie'):
                    patches['accepted_main_delta.trie'] = r.id

        return patches

    def get_task_url(self):
        task = channel.sandbox.get_task(self.id)
        return task.url if task else ''

    def create_branch(self):
        with Key(self, 'robot-ontodb', 'robot-ontodb-ssh-key'):
            dst = SvnWizard(self.ctx[ArcadiaPathParam.name]).branch()
            if Arcadia.check(dst):
                return

            task_url = self.get_task_url()
            if task_url:
                task_url = '. Task: ' + task_url

            Arcadia.copy(
                src=SvnWizard(self.ctx[SourceBranchParam.name]).branch(),
                dst=dst,
                message='Creating new branch' + task_url,
                parents=True,
                user='robot-ontodb',
            )

    @staticmethod
    def check_task(name, task_id):
        task = channel.sandbox.get_task(task_id)
        if not task:
            raise Exception('No {} with id {}'.format(name, task_id))

        if not task.is_done():
            utils.check_tasks_to_finish_correctly(task_ids_list=task.id)

        if not task.is_ok():
            raise Exception('{} {} is not OK'.format(name, task.id))

    def on_execute(self):
        self.add_bugbanner(bugbanner.Banners.EntitySearch, component="entitysearch_releases")

        if '/trunk' in self.ctx[ArcadiaPathParam.name]:
            raise SandboxTaskFailureError("Dont't want to fix trunk")

        if self.ctx[CreateBranchParam.name] and not self.ctx.get(BRANCH_CREATED_MARK):
            self.ctx[BRANCH_CREATED_MARK] = 1
            self.create_branch()

        self.check_task(ConvertedDbTaskParam.name, self.ctx[ConvertedDbTaskParam.name])

        if self.ctx[ConvertedAcceptedMainDeltaTaskParam.name]:
            self.check_task(ConvertedAcceptedMainDeltaTaskParam.name, self.ctx[ConvertedAcceptedMainDeltaTaskParam.name])

        shard_data = self.abs_path('shard_data')
        Arcadia.checkout(SvnWizard(self.ctx[ArcadiaPathParam.name]).entitysearch_data(), shard_data)
        ya_make_path = os.path.join(shard_data, 'ya.make')
        patches = self.get_patches()
        self.patch_ya_make(ya_make_path, patches)
        external_ya_make_path = os.path.join(shard_data, 'external', 'ya.make')
        self.patch_ya_make(external_ya_make_path, patches)
        with Key(self, 'robot-ontodb', 'robot-ontodb-ssh-key'):
            Arcadia.commit(shard_data, 'autoupdating ya.make (no merge)', user='robot-ontodb')


__Task__ = EntitySearchUpdateCmakeResources
