# coding: utf-8

import logging
import random
import string
import os

import sandbox.common.types.resource as ctr

from sandbox.sandboxsdk import task
from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk import process
from sandbox.sandboxsdk import paths
from sandbox.sandboxsdk.channel import channel
from sandbox.projects import resource_types


def gen_temp_name(size=10):
    return ''.join(random.choice(string.ascii_lowercase + string.digits) for _ in xrange(size))


class ExecuteWithPGO(task.SandboxTask):

    class ResourcesBlock(parameters.SandboxInfoParameter):
        description = 'Resources parameters'

    class DependentResource(parameters.LastReleasedResource):
        name = 'dependent_resource_id'
        description = 'Dependent resource'
        state = (ctr.State.READY, ctr.State.NOT_READY)
        required = True

    class ShellExecuteBlock(parameters.SandboxInfoParameter):
        description = 'Shell parameters'

    class BinaryPath(parameters.SandboxStringParameter):
        name = 'binary_path'
        description = 'Binary path in resource'
        required = True

    class ExecutionArgs(parameters.ListRepeater, parameters.SandboxStringParameter):
        name = 'execution_args'
        description = 'Execution args'

    class ExecutionTimeout(parameters.SandboxIntegerParameter):
        name = 'execution_timeout'
        description = 'Execution timeout( in seconds )'
        default_value = 600
        required = True

    class ProfileBlock(parameters.SandboxInfoParameter):
        description = 'PGO profile'

    class ProfileName(parameters.SandboxStringParameter):
        description = 'PGO profile name'
        name = 'pgo_profile_name'
        default_value = 'default.profraw'
        required = True

    type = 'TEST_EXECUTE_WITH_PGO_TASK'

    input_parameters = [ResourcesBlock, DependentResource, ShellExecuteBlock, BinaryPath, ExecutionArgs, ExecutionTimeout, ProfileBlock, ProfileName]

    def on_execute(self):
        logging.debug('Start executing task')
        logging.debug('My parametrs are:\n' +
                      ''.join(('%s:\t%s\n' % (c.description, self.ctx.get(c.name))) if self.ctx.get(c.name) else ''
                              for c in (self.input_parameters) if not isinstance(c, parameters.SandboxInfoParameter))
                      )
        resource_path = channel.sandbox.get_resource(self.ctx['dependent_resource_id']).path
        temp_dir_name = gen_temp_name()
        paths.make_folder(temp_dir_name)
        temp_dir_name = os.path.abspath(temp_dir_name)
        binary_path = os.path.join(resource_path, self.ctx['binary_path'])
        logging.info('Executing binary: %s' % binary_path)
        profile_name = os.path.join(temp_dir_name, '%p-' + self.ctx['pgo_profile_name'])
        env = os.environ.copy()
        env['LLVM_PROFILE_FILE'] = profile_name
        p = process.run_process(
            [binary_path] + self.ctx['execution_args'],
            environment=env,
            check=False,
        )
        process.check_process_timeout(p, self.ctx['execution_timeout'], timeout_sleep=1)
        logging.info('Cmd return code is: %s', p.returncode)
        logging.info('Uploading profiles')
        for prof in paths.list_dir(temp_dir_name, files_only=True, abs_path=True):
            self._create_resource('PGO profile ' + os.path.basename(prof), prof, resource_types.PGO_PROFILE_RESOURCE)


__Task__ = ExecuteWithPGO
