# -*- coding: utf-8 -*-

import sys
import logging

import sandbox.common.types.client as ctc

from sandbox.projects import resource_types
from sandbox.projects.ab_testing.resource_types import EXP_DAEMON_EXECUTABLE
from sandbox.projects.ab_testing.resource_types import BROWSER_XML_RESOURCE

from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.parameters import LastReleasedResource
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.sandboxsdk.errors import SandboxTaskFailureError


class ExpDaemonExecutable(LastReleasedResource):
    name = 'exp_daemon'
    description = 'exp daemon executable'
    resource_type = EXP_DAEMON_EXECUTABLE


class BrowserXml(LastReleasedResource):
    name = 'browser.xml'
    description = 'browser.xml file for uatraits'
    resource_type = BROWSER_XML_RESOURCE


class ExpConfig(LastReleasedResource):
    name = 'config'
    description = 'Experiments config'
    resource_type = resource_types.EXPERIMENTS_CONFIG


class ExperimentsConfigCheck(SandboxTask):
    type = 'EXPERIMENTS_CONFIG_CHECK'

    cores = 1
    ram = 4048
    execution_space = 4096  # 1 Gb

    input_parameters = [
        ExpDaemonExecutable,
        ExpConfig,
        BrowserXml,
    ]

    client_tags = ctc.Tag.Group.LINUX

    def arcadia_info(self):
        return None, "Experiments config #{}".format(self.ctx.get('version', 'none')), None

    def get_exp_config_checker(self):
        logging.info("Fetching checker script.")
        exp_config_checker_svn_url = (
            'arcadia:/arc/trunk/arcadia/quality/ab_testing/scripts/exp_config_checker/exp_config_checker.py'
        )
        exp_config_checker_path = self.abs_path('exp_config_checker.py')
        Arcadia.export(exp_config_checker_svn_url, exp_config_checker_path)
        return exp_config_checker_path

    def on_execute(self):
        exp_config_checker_path = self.get_exp_config_checker()
        exp_daemon = channel.task.sync_resource(self.ctx[ExpDaemonExecutable.name])
        config = channel.task.sync_resource(self.ctx[ExpConfig.name])
        browser_xml = channel.task.sync_resource(self.ctx[BrowserXml.name])

        if config.endswith('configs.tgz'):
            self._subprocess('tar -xzf {}'.format(config), check=True, wait=True, log_prefix='extract_tar')
            configs_to_check = ['experiments_config.tsv', ]  # 'experiments_config_excomer.tsv', ]
        else:
            configs_to_check = [config, ]

        success = True

        for config_to_check in configs_to_check:
            logging.info('Running checker script for %s', config_to_check)
            success = success and run_process(
                [sys.executable, exp_config_checker_path, "-b", exp_daemon, "-c", config_to_check, "-u", browser_xml],
                log_prefix="checker",
                work_dir=self.path()
            ).returncode == 0

            if not success:
                raise SandboxTaskFailureError('Failed to check config!')


__Task__ = ExperimentsConfigCheck
