# -*- coding: utf-8 -*-

import six
import base64
import codecs
import json
import logging
import time
import urllib2
import urllib
import socket

import sandbox.common.types.misc as ctm

from sandbox.projects import resource_types
from sandbox.projects.common.nanny import nanny
from sandbox.projects.common import utils

from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.parameters import SandboxStringParameter
from sandbox.sandboxsdk.parameters import SandboxBoolParameter
from sandbox.sandboxsdk.parameters import LastReleasedResource
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.svn import Svn

import validator


class ConfigUrlParameter(SandboxStringParameter):
    name = 'config_url'
    description = 'Config URL'
    default_value = "http://ab.yandex-team.ru/config/13/tag/production/usersplit"


class ConfigMergerUrlParameter(SandboxStringParameter):
    name = 'config_merger_url'
    description = 'Config for the Merger URL'
    default_value = ''  # "http://ab.yandex-team.ru/config/13/tag/production/usersplit/excomer"


class GeodataUrlParameter(SandboxStringParameter):
    name = 'geodata_url'
    description = 'URL to geodata5.bin'
    default_value = "rsync://veles.yandex.ru/Berkanavt/geodata/geodata5.bin"


class IPREGUrlParameter(SandboxStringParameter):
    name = 'ipreg_url'
    description = 'URL to IPREG'
    default_value = "rsync://veles.yandex.ru/Berkanavt/geodata/ipreg/IPv6"


class AppendGeojuiceParameter(SandboxBoolParameter):
    name = 'append_geojuice'
    description = 'Cook geojuice'
    default_value = True


class NeedBalancerCheckParameter(SandboxBoolParameter):
    name = 'need_balancer_check'
    description = 'Balancer check is needed'
    default_value = True


class AdminkaUtilsGeoParameter(LastReleasedResource):
    name = 'utils_geo'
    description = 'Utility extract_geo_tree'
    resource_type = resource_types.EXPERIMENTS_ADMINKA_UTILS_GEO


class AdminkaUtilsSqueezeParameter(LastReleasedResource):
    name = 'utils_squeeze'
    description = 'Utility squeeze_geobase.py'
    resource_type = resource_types.EXPERIMENTS_ADMINKA_UTILS_SQUEEZE


class IsSinkConfigParameter(SandboxBoolParameter):
    name = 'is_sink_config'
    description = 'Sink config (testing)'
    default_value = False


class AnalyzethisIpregParameter(LastReleasedResource):
    name = 'analyzethis_ipreg'
    description = 'IPREG.analyzethis (pseudo)'
    resource_type = resource_types.ANALYZETHIS_IPREG


class ExperimentsConfigUpdate(nanny.ReleaseToNannyTask, SandboxTask):
    type = 'EXPERIMENTS_CONFIG_UPDATE'
    dns = ctm.DnsType.DNS64

    cores = 1
    ram = 4096
    execution_space = 5120  # 5 Gb for geobase

    input_parameters = [
        ConfigUrlParameter,
        ConfigMergerUrlParameter,
        GeodataUrlParameter,
        IPREGUrlParameter,
        AppendGeojuiceParameter,
        NeedBalancerCheckParameter,
        AdminkaUtilsGeoParameter,
        AdminkaUtilsSqueezeParameter,
        IsSinkConfigParameter,
        AnalyzethisIpregParameter,
    ]

    def arcadia_info(self):
        return None, "Experiments config #{}".format(self.ctx.get('version', 'none')), None

    def on_release(self, additional_parameters):
        logging.info("release config")
        version = self.ctx.get('version')
        version_old = self._get_current_version()
        if self.ctx[IsSinkConfigParameter.name]:
            config_type = ' sink '
        else:
            config_type = ' '
        comment = 'Update experiments{}config to version #{}'.format(config_type, version)
        deploy_comment = None
        if version_old is not None:
            deploy_comment = (
                "Diff http://ab.yandex-team.ru/config/13/diff/{}/{} \n"
                "Instruction "
                "https://wiki.yandex-team.ru/serp/experiments/usersplit/searchmon".format(
                    version_old, version
                )
            )

        logging.debug("COMMENT = {}".format(comment))
        logging.debug("DEPLOY COMMENT = {}".format(deploy_comment))
        additional_parameters['release_subject'] = comment
        additional_parameters['release_comments'] = deploy_comment
        nanny.ReleaseToNannyTask.on_release(self, additional_parameters)

    def _get_current_version(self):
        url = "http://ab.yandex-team.ru/api/expstorage/production/json"
        try:
            req = urllib2.Request(url)
            resp = urllib2.urlopen(req, timeout=10)
            config = resp.read()
            version = int(json.loads(config)["version"])
            return version
        except Exception as e:
            logging.warning("ERROR getting expstorage version: {}".format(str(e)))
        return None

    def on_execute(self):
        if 'config_resource_id' in self.ctx:
            utils.check_subtasks_fails(stop_on_broken_children=False)
        else:
            config = self._download_config(self.ctx['config_url'])
            config_without_geo = config

            logging.debug("Current expstorage version: {}".format(self._get_current_version()))

            version, nodes_full = validator.parse_config(config)
            self.ctx['version'] = version
            validator.validate_nodes(nodes_full)

            if self.ctx['config_merger_url']:
                config_merger = self._download_config(self.ctx['config_merger_url'])
                _, nodes_excomer = validator.parse_config(config_merger)
                validator.validate_nodes(nodes_excomer)
            else:
                config_merger = None

            if self.ctx[AppendGeojuiceParameter.name]:
                regions = validator.extract_nodes_regions(nodes_full)
                logging.info("REGIONS: {}".format(str(regions)))

                path_geodata = self._prepare_geobase()
                geojuice, geojuice_regid = self._extract_geojuice(regions, path_geodata)

                if not config[-1] == "\n":
                    config += "\n"
                config += geojuice + "\n" + geojuice_regid

                if config_merger:
                    if not config_merger[-1] == "\n":
                        config_merger += "\n"
                    config_merger += geojuice + "\n" + geojuice_regid

            config_id = self._make_resource(config, config_merger, config_without_geo)
            if self.ctx[NeedBalancerCheckParameter.name]:
                self._check_config(config_id)
                utils.wait_all_subtasks_stop()

    def _download_config(self, config_url):
        logging.info('Fetching config from url %s' % (config_url))
        for i in range(10):
            logging.info('fetch attempt %d' % (i + 1))
            try:
                # try to set timeout here due to USEREXP-5312
                return urllib2.urlopen(config_url, timeout=120).read()
            except urllib2.URLError as e:
                logging.warning("error: %s" % (str(e)))
            except socket.timeout as e:
                logging.warning("timeout: %s" % (str(e)))
            time.sleep(10)
        raise Exception("FAILED to fetch config")

    def _make_resource(self, config_text, config_merger_text=None, config_without_geo_text=None):
        logging.info('Writing config to file')
        config_filepath = self.abs_path('experiments_config.tsv')
        with codecs.open(config_filepath, 'w', 'utf-8') as f:
            f.write(config_text)

        logging.info('Validating config file')
        validator.validate_config_file(config_filepath)

        if config_merger_text or config_without_geo_text:
            configs_tar_filepath = self.abs_path('configs.tgz')
            cmd = 'tar -czf {} experiments_config.tsv'.format(configs_tar_filepath)

            if config_merger_text:
                config_merger_filepath = self.abs_path('experiments_config_excomer.tsv')
                with codecs.open(config_merger_filepath, 'w', 'utf-8') as f:
                    f.write(config_merger_text)
                logging.info('Validating config merger file')
                validator.validate_config_file(config_merger_filepath)
                cmd += ' experiments_config_excomer.tsv'

            if config_without_geo_text:
                config_without_geo_filepath = self.abs_path('experiments_config_without_geo.tsv')
                with codecs.open(config_without_geo_filepath, 'w', 'utf-8') as f:
                    f.write(config_without_geo_text)
                cmd += ' experiments_config_without_geo.tsv'

            self._subprocess(cmd, check=True, wait=True, log_prefix='build_tar')

            resource_path = configs_tar_filepath
        else:
            resource_path = config_filepath

        logging.info('Creating resource')
        if self.ctx[IsSinkConfigParameter.name]:
            resource_type = resource_types.EXPERIMENTS_CONFIG_SINK
            name_suffix = 'sink_'
        else:
            resource_type = resource_types.EXPERIMENTS_CONFIG
            name_suffix = ''
        resource = self._create_resource('experiments_config_{}{}'.format(name_suffix, self.ctx['version']),
                                         resource_path,
                                         resource_type)
        self.ctx['config_resource_id'] = resource.id
        self.mark_resource_ready(resource)
        return resource.id

    def _extract_config_regions(self, config):
        lines = config.split("\n")

        num_params = int(lines[0])
        num_nodes = int(lines[num_params + 1])

        nodes = lines[(num_params + 2):(num_params + 2 + num_nodes)]

        regions = set()
        for node in nodes:
            p = node.split("\t")
            if p[0] == 'r':
                region = int(p[2])
                regions.add(region)

        regions = sorted(regions)
        logging.info("REGIONS: {}".format(str(regions)))
        return regions

    def _extract_geojuice(self, regions, path_geodata):
        path_squeezer = channel.task.sync_resource(self.ctx[AdminkaUtilsSqueezeParameter.name])

        url_ipreg = self.ctx[IPREGUrlParameter.name]

        ipregs = (
            ['--ipreg_detailed', 'IPREG.detailed'],
            ['--ipreg_yandex', 'IPREG.yandex'],
        )
        extra_params = []

        for ipreg in ipregs:
            path = self.path(ipreg[1])
            url = (url_ipreg +
                   ("" if url_ipreg[-1] == "/" else "/") +
                   ipreg[1])
            self._download_file_attack(url, path, attempts=5)

            extra_params.append(ipreg[0])
            extra_params.append(path)

        path_geojuice = self.path('geojuice.bin')
        path_geojuice_regid = self.path('geojuice_regid.bin')

        path_ipreg_analyzethis = channel.task.sync_resource(self.ctx[AnalyzethisIpregParameter.name])

        run_process(
            ['python', path_squeezer,
             '--ipv6',
             '-o', path_geojuice,
             '--output_regid', path_geojuice_regid,
             '--geobase_tsv', path_geodata,
             '--ipreg_analyzethis', path_ipreg_analyzethis] +
            extra_params +
            [','.join([str(x) for x in regions])],
            log_prefix='squeeze-juice',
            work_dir=self.path()
        )

        with open(path_geojuice, 'r') as f:
            geojuice = f.read()

        with open(path_geojuice_regid, 'r') as f:
            geojuice_regid = f.read()

        geojuice_b64 = base64.b64encode(geojuice)
        geojuice_regid_b64 = base64.b64encode(geojuice_regid)
        return geojuice_b64, geojuice_regid_b64

    def _prepare_geobase(self):
        path_geodata = self.path("geodata5.bin")
        path_geodata_tsv = self.path("geodata5.tsv")

        url_geodata = self.ctx[GeodataUrlParameter.name]

        logging.info("Extract geodata {}".format(url_geodata))
        self._download_file_attack(url_geodata, path_geodata, attempts=5)

        path_extracter = channel.task.sync_resource(self.ctx[AdminkaUtilsGeoParameter.name])

        run_process(
            [path_extracter,
             "-g", path_geodata,
             "-o", path_geodata_tsv],
            log_prefix="extract-geobase",
            work_dir=self.path()
        )

        return path_geodata_tsv

    def _download_file_attack(self, url, path, attempts=1):
        logging.info("DOWNLOAD ATTACK {} to {}".format(url, path))

        while attempts > 0:
            try:
                return self._download_file(url, path)
            except Exception as e:
                logging.debug(" -- download failed: {}".format(six.text_type(e)))
                if attempts == 1:
                    raise
            attempts -= 1

    def _download_file(self, url, path):
        logging.info("DOWNLOAD {} to {}".format(url, path))
        if url[:8] == "rsync://":
            run_process(
                ['rsync', url, path],
                log_prefix="rsync-download",
                work_dir=self.path()
            )
        elif url[:10] == "svn+ssh://":
            Svn.export(url, path)
        else:
            urllib.urlretrieve(url, path)

    def _check_config(self, config_id):
        logging.info("checking experiments config")
        self.create_subtask(
            task_type="EXPERIMENTS_CONFIG_CHECK",
            description="Check experiments config for EXPERIMENTS_CONFIG_UPDATE task #{}".format(self.id),
            input_parameters={"config": config_id}
        )


__Task__ = ExperimentsConfigUpdate
