# -*- coding: utf-8 -*-

import os

import logging

import sandbox.common.types.client as ctc

from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.task import SandboxTask
import sandbox.sandboxsdk.parameters as sp
from sandbox.projects import resource_types
import sandbox.sandboxsdk.process as sbproc

from sandbox.projects.common import apihelpers
from sandbox.projects.common import utils
import sandbox.projects.common.constants as consts
from sandbox.sandboxsdk.svn import Arcadia


class CheckoutArcadiaFromUrlParam(sp.SandboxStringParameter):
    name = consts.ARCADIA_URL_KEY
    description = 'Svn url for arcadia'
    default_value = Arcadia.trunk_url()
    required = True


class IdxQdumpBinaryResourceParam(sp.LastReleasedResource):
    # to simplify the task debugging
    name = 'idx_qdump_binary_resource_id'
    description = 'idx_qdump binary for quick task debug without build'
    resource_type = resource_types.IDX_QDUMP_EXECUTABLE
    required = False


class SearchDatabaseParam(sp.LastReleasedResource):
    # to debug the task create a fake "database" folder with 2 panther index files
    name = 'search_database_resource_id'
    description = 'shard with panther index'
    resource_type = resource_types.SEARCH_DATABASE
    required = True


class WizardedRequestsParam(sp.LastReleasedResource):
    # to debug the task create a small requests file
    name = 'wizarded_reqs_resource_id'
    description = 'wizarded requests'
    resource_type = resource_types.WIZARDED_POOL_REQUESTS
    required = True


class ExtractPantherGuts(SandboxTask):
    """
    **Извлечь кишки из указанного пантерного индекса по переколдованным запросам**
    """

    type = 'EXTRACT_PANTHER_GUTS'

    client_tags = ctc.Tag.Group.LINUX

    def __init__(self, task_id=0):
        SandboxTask.__init__(self, task_id)
        self.guts_dir = 'tries_bundle'
        self.guts_arc = '{}.tar.gz'.format(self.guts_dir)

    input_parameters = (
        CheckoutArcadiaFromUrlParam,
        IdxQdumpBinaryResourceParam,
        SearchDatabaseParam,
        WizardedRequestsParam,
    )

    def _build_binaries(self):
        builds_data = {
            'idx_qdump_build_id':
                ('linux', 'build_idx_qdump'),
        }
        wait_builds = []
        for build_key in builds_data:
            if not self.ctx.get(build_key):
                self.ctx[build_key] = self._schedule_build(*builds_data[build_key])
                wait_builds.append(self.ctx[build_key])
        if len(wait_builds):
            utils.wait_all_subtasks_stop()
        else:
            utils.check_subtasks_fails(stop_on_broken_children=True)
        self.ctx['idx_qdump_binary_resource_id'] = apihelpers.get_task_resource_id(self.ctx['idx_qdump_build_id'], 'IDX_QDUMP_EXECUTABLE')

    def run_idx_qdump(self):
        binary_id = self.ctx['idx_qdump_binary_resource_id']
        binary_path = channel.task.sync_resource(binary_id)

        shard_id = self.ctx['search_database_resource_id']
        shard_path = channel.task.sync_resource(shard_id)

        reqs_id = self.ctx['wizarded_reqs_resource_id']
        reqs_path = channel.task.sync_resource(reqs_id)

        os.mkdir(self.guts_dir)

        idx_prefix = os.path.join(shard_path, 'indexpanther.')

        cmd = [binary_path, idx_prefix, reqs_path, self.guts_dir]
        sbproc.run_process(cmd, log_prefix='idx_qdump_log')
        sbproc.run_process(['tar -zcf {} {}'.format(self.guts_arc, self.guts_dir)], shell=True, log_prefix='tar_log')

    def on_enqueue(self):
        SandboxTask.on_enqueue(self)
        self.ctx['panther_guts_resource_id'] = self._create_resource('extracted guts', self.guts_arc, resource_types.PANTHER_GUTS, arch=self.arch).id

    def on_execute(self):
        if not self.ctx['idx_qdump_binary_resource_id']:
            self._build_binaries()
        self.run_idx_qdump()

    def _schedule_build(self, arch, build_target):
        description = "{} {} {}".format(self.descr, build_target, arch)
        logging.info('description ' + description)
        sub_ctx = {
            consts.ARCADIA_URL_KEY: self.ctx[consts.ARCADIA_URL_KEY],
            build_target: True,
        }
        task = self.create_subtask(task_type='BUILD_SEARCH', description=description,
                                      input_parameters=sub_ctx, arch=arch, important=self.important)
        return task.id


__Task__ = ExtractPantherGuts
