# -*- coding: utf-8 -*-

from sandbox.projects import resource_types
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk import parameters as sp

from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common import utils
from sandbox.projects.common.search import bugbanner
from sandbox.projects.common.base_search_quality import priemka
from sandbox.projects.common.base_search_quality.response_diff_task import FailOnDiffParameter

from sandbox.projects import MixQueriesExperimentsRegions as mq


def get_config():
    return [
        get_config_by_name(name)
        for name in priemka.get_basesearch_production_shard_tag_names()
    ]


def get_config_by_name(n=""):
    class Config(sp.LastReleasedResource):
        name = 'new_config_{}_res_id'.format(n)
        description = '{} config'.format(n)
        resource_type = [resource_types.SEARCH_CONFIG]

    return Config


class NewConfigs(sp.SandboxBoolParameter):
    name = 'new_configs'
    description = 'New configs'
    sub_fields = {'true': [config.name for config in get_config()]}
    default_value = False


class TestConcurrentRelevance(sp.SandboxBoolParameter):
    name = 'test_concurrent_relevance'
    description = 'Test concurrent relevance'
    default_value = False


class UseBinaryResponses(sp.SandboxBoolParameter):
    name = 'use_binary_responses'
    description = 'Use responses in binary format'
    default_value = False


class NewCgiParam(sp.SandboxStringParameter):
    name = 'new_cgi_param'
    description = 'New cgi params (must be encoded, may contain multiple params divided by \';\')'
    default_value = ''
    group = "Queries generation setup"


class ChildTaskPriority(sp.SandboxIntegerParameter):
    name = 'child_tasks_priority'
    description = 'Child task priority'
    default_value = 100


class DevelopmentMode(sp.SandboxBoolParameter):
    name = 'development_mode'
    description = 'Development mode'
    default_value = False


other_res_source = priemka.get_source_for_other_resources()


class FastBasesearchAcceptance(bugbanner.BugBannerTask):
    """
        Задача определяет, для каких типов шардов в sandbox есть данные, загруженные ранее.
        Для каждого типа шарда делает следующее:

        * Подготавливает запросы с помощью запуска subtask'а MIX_QUERIES_EXPERIMENTS_REGIONS.
        * Получает выдачу по двум бинарникам путем запуска subtask'ов GET_BASESEARCH_RESPONSES
        * Сравнивает две выдачи путём запуска task'а COMPARE_BASESEARCH_RESPONSES
    """
    type = 'FAST_BASESEARCH_ACCEPTANCE'

    input_parameters = [
        priemka.ExecutableSource1,
        priemka.Executable1,
        priemka.ExecutableSource2,
        priemka.Executable2,
        priemka.DynamicModelsSource1,
        priemka.DynamicModels1,
        priemka.DynamicModelsSource2,
        priemka.DynamicModels2,
        other_res_source,
    ] + priemka.get_other_existing_res() + [
        priemka.GetAllFactors,
        priemka.PoliteMode,
        # priemka.IgnoreGotError,
        NewCgiParam,
        UseBinaryResponses,
        ChildTaskPriority,
        DevelopmentMode,
        TestConcurrentRelevance,
        NewConfigs,
        FailOnDiffParameter,
    ] + get_config() + mq.exp_setup_params.params + mq.queries_gen_params.params
    execution_space = 256  # 256 Mb

    def initCtx(self):
        priemka.reset_resources_ids(self)

    def on_execute(self):
        self.add_bugbanner(bugbanner.Banners.WebBaseSearch)

        if "compare_task_ids" not in self.ctx:
            self.ctx["compare_task_ids"] = self._start_child_tasks()
            utils.wait_all_subtasks_stop()
        else:
            utils.check_subtasks_fails(stop_on_broken_children=True)

            if self.ctx[TestConcurrentRelevance.name]:
                for task in self.get_compare_tasks():
                    eh.ensure(task.ctx["compare_result"], "concurrent relevance changes basesearch responses")
            else:
                self.ctx['diffs'] = [
                    task.ctx['shard_tag']
                    for task in self.get_compare_tasks() if task.is_finished() and (not task.ctx["compare_result"])
                ]
            self._set_attr_for_formulas()

    def _start_child_tasks(self):
        res_by_shard = priemka.get_shard_tags_and_resources(dev_mode=self.ctx[DevelopmentMode.name])
        self.ctx['patched_queries'] = {}
        return [self._start_tasks_for_shard_tag(shard_tag, r) for shard_tag, r in res_by_shard.iteritems()]

    def _start_tasks_for_shard_tag(self, shard_tag, resources):
        descr = "patched queries for shard tags {}".format(shard_tag)
        resources["patched_queries_resource_id"] = self._prepare_queries(resources["queries"].id, descr)

        self.ctx['patched_queries'][shard_tag] = resources["patched_queries_resource_id"]

        exe1 = priemka.get_basesearch_exe(self.ctx, 1)
        exe2 = priemka.get_basesearch_exe(self.ctx, 2)

        responses1_res_id = self._get_basesearch_responses(
            exe1, resources, priemka.get_models_archive(self.ctx, 1),
            concurrent_relevance=False,  # без параллельной релевантности (relevance_threads == 0)
            task_description="responses for {} for first basesearch and models".format(shard_tag),
        )

        if self.ctx[NewConfigs.name]:
            resources["config"] = self._read_resource(self.ctx["new_config_{}_res_id".format(shard_tag)], sync=False)
        responses2_res_id = self._get_basesearch_responses(
            exe2, resources, priemka.get_models_archive(self.ctx, 2),
            concurrent_relevance=self.ctx.get(TestConcurrentRelevance.name),  # для теста CONCURRENT_RELEVANCE
            task_description="responses for {} for second basesearch/config and models".format(shard_tag),
        )

        return self._start_compare_task(responses1_res_id, responses2_res_id, shard_tag)

    def _prepare_queries(self, queries_source, task_description):
        exp = mq.exp_setup_params
        q_gen = mq.queries_gen_params
        sub_ctx = {
            mq.Queries.name: queries_source,
            exp.ExpsChanged.name: self.ctx[exp.ExpsChanged.name],
            exp.SnipExpsChanged.name: self.ctx[exp.SnipExpsChanged.name],
            q_gen.NumOfSearchQueries.name: self.ctx[q_gen.NumOfSearchQueries.name],
            q_gen.NumOfXmlSearchQueries.name: self.ctx[q_gen.NumOfXmlSearchQueries.name],
            q_gen.NumSnippetQueries.name: self.ctx[q_gen.NumSnippetQueries.name],
            q_gen.QueriesWithModifiedExp.name: self.ctx[q_gen.QueriesWithModifiedExp.name],
            q_gen.QueriesWithModifiedReg.name: self.ctx[q_gen.QueriesWithModifiedReg.name],
            q_gen.QueriesVsRegRatio.name: self.ctx[q_gen.QueriesVsRegRatio.name],
            q_gen.FailOnLowAmountOfQueries.name: (
                (not self.ctx[TestConcurrentRelevance.name]) and self.ctx[q_gen.FailOnLowAmountOfQueries.name]
            )
        }

        subtask = self.create_subtask(
            task_type='MIX_QUERIES_EXPERIMENTS_REGIONS',
            input_parameters=sub_ctx,
            description=task_description,
            priority=self.ctx[ChildTaskPriority.name]
        )

        queries_resource_id = int(subtask.ctx['out_resource_id'])

        new_cgi_param = self.ctx.get(NewCgiParam.name)
        if new_cgi_param:
            sub_ctx = {
                "queries_resource_id": queries_resource_id,
                "new_cgi_param": new_cgi_param
            }

            subtask = self.create_subtask(
                task_type='PATCH_QUERIES',
                input_parameters=sub_ctx,
                description=task_description,
                priority=self.ctx[ChildTaskPriority.name]
            )

            queries_resource_id = subtask.ctx['out_resource_id']

        return queries_resource_id

    def _get_basesearch_responses(
        self,
        exe_res_id,
        resources,
        dynamic_models_archive_res_id,
        concurrent_relevance,
        task_description,
    ):

        sub_ctx = {
            "basesearch_type": "basesearch",
            "executable_resource_id": exe_res_id,
            "config_resource_id": resources["config"].id,
            "search_database_resource_id": resources["db"].id,
            "queries_resource_id": resources["patched_queries_resource_id"],
            "models_archive_resource_id": dynamic_models_archive_res_id,
            "min_output_size": 6000000,
            "relevance_threads": 5 if concurrent_relevance else 0,
            "get_all_factors": self.ctx[priemka.GetAllFactors.name],
            "polite_mode": self.ctx[priemka.PoliteMode.name],
            # SEARCH-2084
            "ignore_got_error": True,  # self.ctx[priemka.IgnoreGotError.name],
            "max_fail_rate": 0.2,  # allow some 'site:' queries
            "get_responses_from_binary": utils.get_or_default(self.ctx, UseBinaryResponses),
        }

        subtask = self.create_subtask(
            task_type='GET_BASESEARCH_RESPONSES',
            input_parameters=sub_ctx,
            description=task_description,
            priority=self.ctx[ChildTaskPriority.name],
            arch='linux',
            model='e5-2650',  # SEARCH-8286
        )

        return int(subtask.ctx["out_resource_id"])

    def _start_compare_task(self, responses1_res_id, responses2_res_id, shard_tag):
        sub_ctx = {
            "basesearch_responses1_resource_id": responses1_res_id,
            "basesearch_responses2_resource_id": responses2_res_id,
            "shard_tag": shard_tag,
            "queries_per_file": 100,
            "process_count": 1,
            "use_multiprocessing": False,  # SEARCH-2287
            FailOnDiffParameter.name: utils.get_or_default(self.ctx, FailOnDiffParameter)
        }
        cmp_task_type = "COMPARE_BASESEARCH_RESPONSES"
        if utils.get_or_default(self.ctx, UseBinaryResponses):
            cmp_task_type = "COMPARE_BINARY_RESPONSES"
        return self.create_subtask(
            task_type=cmp_task_type,
            input_parameters=sub_ctx,
            description="diff for shard tag {}".format(shard_tag),
            priority=self.ctx[ChildTaskPriority.name]
        ).id

    def get_compare_tasks(self):
        if "compare_task_ids" not in self.ctx:
            return []

        tasks = [channel.sandbox.get_task(_id) for _id in self.ctx["compare_task_ids"]]
        tasks = [t for t in tasks if not t.is_deleted()]
        return tasks

    def _set_attr_for_formulas(self):
        """https://st.yandex-team.ru/SEARCH-1916#1467157584000"""
        formulas_to_release = self.ctx.get('dynamic_models2_res_id')
        if 'web_formulas_acceptance' in self.ctx and formulas_to_release:
            utils.set_resource_attributes(
                formulas_to_release,
                {'web_formulas_acceptance': self.ctx['web_formulas_acceptance']}
            )


__Task__ = FastBasesearchAcceptance
