# -*- coding: utf-8 -*-

from sandbox.projects import resource_types
from sandbox.projects.common.search.eventlog.eventlog import filter_eventlog

from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk import parameters


class EvlogdumpExecutableParameter(parameters.ResourceSelector):
    name = 'evlogdump_executable_resource_id'
    description = 'Evlogdump executable'
    resource_type = resource_types.EVLOGDUMP_EXECUTABLE
    required = True


class EventlogParameter(parameters.ResourceSelector):
    name = 'eventlog_resource_id'
    description = 'Eventlog'
    resource_type = resource_types.EVENTLOG_DUMP
    required = True


class Filter(parameters.SandboxStringParameter):
    name = 'filter'
    description = 'Choose filter'
    choices = [
        ('turkish', 'turkish'),
        ('custom', 'custom')
    ]
    sub_fields = {
        'custom': ['evlogdump_options', 'custom_filter']
    }
    default_value = 'turkish'


class EvlogdumpOpts(parameters.SandboxStringParameter):
    name = 'evlogdump_options'
    description = 'Evlogdump options'
    default_value = '-o'


class CustomFilter(parameters.SandboxStringParameter):
    name = 'custom_filter'
    description = 'Custom filter'
    required = True


class AddQuestionMark(parameters.SandboxBoolParameter):
    name = 'add_question_mark'
    description = 'Add question mark in the beginning of each string'


class FilterEventlog(SandboxTask):
    """
        **Описание**

            Фильтрует эвентлог, используя произвольно заданный фильтр

        **Ресурсы**

            *Обязательные*

            * **Evlogdump executable**
            * **Eventlog**

        **Параметры**

        * **Choose filter** - выбор фильтра из списка:

        1. turkish - оставляет только турецкие запросы
        2. custom - фильтр с произвольными параметрами, имеет подполя: (Evlogdump options и Custom filter)

        * **Add question mark in the beginning of each string** (`add_question_mark`) - так как чаще всего таск применяется для фильтрования определенных запросов,
          но в эвентлоге запросы приведены без знака вопроса в начале, эта опция позволяет добавить знак вопроса в начало каждой отфильтрованной строки
    """
    type = 'FILTER_EVENTLOG'

    input_parameters = (
        EvlogdumpExecutableParameter,
        EventlogParameter,
        Filter,
        EvlogdumpOpts,
        CustomFilter,
        AddQuestionMark
    )

    def on_execute(self):
        self._set_filter_opts()

        evlogdump_exe_path = self.sync_resource(self.ctx[EvlogdumpExecutableParameter.name])
        eventlog_path = self.sync_resource(self.ctx[EventlogParameter.name])

        out_resource = self.create_resource(
            description='Parsed eventlog',
            resource_path='parsed_eventlog.txt',
            resource_type=resource_types.OTHER_RESOURCE
        )
        filter_eventlog(
            evlogdump_exe_path, eventlog_path,
            filter_command=self.ctx[CustomFilter.name],
            output_path=out_resource.path,
            evlogdump_options=self.ctx[EvlogdumpOpts.name],
        )

        if self.ctx.get(AddQuestionMark.name):
            with open(out_resource.path) as f:
                lines = f.readlines()
            with open(out_resource.path, "w") as f:
                for line in lines:
                    f.write("?" + line)

    def _set_filter_opts(self):
        filter_case = self.ctx.get(Filter.name)
        if filter_case == 'turkish':
            self.ctx[EvlogdumpOpts.name] = "-o -i 287"
            self.ctx[CustomFilter.name] = "grep tld=com.tr | awk '{print $4}'"


__Task__ = FilterEventlog
