# coding=utf-8

import logging
import collections

from sandbox import sdk2

import sandbox.projects.findurl.resources as findurl_res
from sandbox.projects.common import decorators
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common import requests_wrapper
from sandbox.projects.common.nanny import client
from sandbox.projects.common.search import findurl
from sandbox.common.types import task


RELEASE_SOURCES = {
    "arc": "Arcadia (+patch)",
    "task": "Completed YA_PACKAGE task",
}

RELEASE_TARGETS = [
    "findurl",
    "findurl_dev",
]

RELEASE_TARGETS_TO_URLS = {
    "findurl": "findurl.n.yandex-team.ru",
    "findurl_dev": "findurl-dev.n.yandex-team.ru",
    "findurl_priemka": "findurl_priemka.n.yandex-team.ru",
}


class FindUrlRelease(sdk2.Task):

    class Parameters(sdk2.Task.Parameters):

        with sdk2.parameters.RadioGroup("Release source", required=True) as release_source:
            release_source.values["arc"] = RELEASE_SOURCES["arc"]
            release_source.values["task"] = RELEASE_SOURCES["task"]

        with release_source.value['arc']:
            arcadia_url = sdk2.parameters.ArcadiaUrl(
                "Svn url for arcadia (you can add '@<commit_number>')",
                required=True,
                default_value='arcadia:/arc/trunk/arcadia'
            )

            arcadia_patch = sdk2.parameters.String(
                'Apply patch (diff file rbtorrent, paste.y-t.ru link or plain text). Doc: https://nda.ya.ru/3QTTV4',
                default='', multiline=True
            )

        with release_source.value['task']:
            ya_make_task_id = sdk2.parameters.String(
                'Task id of completed YA_PACKAGE task',
                default='', required=True
            )

        with sdk2.parameters.CheckGroup("Release target", required=True) as release_target:
            release_target.choices = [(_, _) for _ in RELEASE_TARGETS]

        run_priemka = sdk2.parameters.Bool("Run priemka first", default=True)
        force = sdk2.parameters.Bool("Force release even if there are running tasks (not recommended)", default=False)

    class Requirements(sdk2.Task.Requirements):
        pass

    class Context(sdk2.Context):
        subtasks_launched = False  # Have subtasks been launched (used to prevent infinite loop after WaitTask handling)
        errors = []  # Errors occurred during execution of subtasks (used in footer)
        messages = []  # Messages about results of execution of subtasks (used in footer)

    def launch_and_wait_subtasks(self):
        self.Context.abc = self.Parameters.arcadia_url
        package_build_task = sdk2.Task["YA_PACKAGE"](
            self,
            description="Child of {}".format(self.id),
            owner=self.owner,
            checkout_arcadia_from_url=self.Parameters.arcadia_url,
            package_type='tarball',
            packages='packages/debian/runtime/webrobot/findurl/pkg.json',
            arcadia_patch=self.Parameters.arcadia_patch,
            resource_type=findurl_res.FindurlPackage.name,
        )
        package_build_task.enqueue()
        self.Context.subtasks_launched = True
        raise sdk2.WaitTask([package_build_task], task.Status.Group.FINISH | task.Status.Group.BREAK, wait_all=True)

    def check_subtask_status(self, subtask):
        if subtask.status in (task.Status.FAILURE, task.Status.EXCEPTION):
            self.Context.errors.append('Subtask {} failed'.format(subtask.id))
        else:
            self.Context.messages.append('Subtask {} have finished successfully'.format(subtask.id))

    @decorators.retries(7, 1, 2)
    def wait_working_tasks(self, rel_target):
        if self.Parameters.force:
            return
        r = requests_wrapper.post(
            'https://{}/tasks/'.format(RELEASE_TARGETS_TO_URLS[rel_target]),
            data={'number': 300},
            verify=False
        )
        if r.text.count('WORKING'):
            raise Exception("There are tasks in WORKING state")

    @staticmethod
    def is_ya_package_in_nanny(nanny_client, rel_target, task_id):
        resources = nanny_client.get_service_resources(rel_target)
        sandbox_files = resources["content"]["sandbox_files"]
        return any(sb_file["task_type"] == "YA_PACKAGE" and sb_file["task_id"] == task_id for sb_file in sandbox_files)

    @decorators.retries(8, 3, 2)
    def nanny_check_deploy(self, nanny_client, rel_target, task_id, old_time_value):
        logging.debug("Checking deploy")
        current_state = nanny_client.get_service_current_state(rel_target)
        if current_state["content"]["summary"]["entered"] == old_time_value:
            raise Exception("Service is still in previous state")
        if current_state["content"]["summary"]["value"] != "ONLINE":
            raise Exception("Service is still not ONLINE")
        c = collections.Counter(snapshot["state"] for snapshot in current_state["content"]["active_snapshots"])
        if not(c["ACTIVE"] == 1 and c["PREPARED"] == 1):
            raise Exception("There should be one ACTIVE and one PREPARED snapshot")

        if FindUrlRelease.is_ya_package_in_nanny(nanny_client, rel_target, task_id):
            return
        raise Exception("New package has not been deployed")

    @staticmethod
    def make_simple_doc(query_text, url, region_id='213'):
        return findurl.FindUrl.Document(
            findurl.FindUrl.Query(
                {'query': {'text': query_text, 'regionId': region_id, 'uid': '', 'device': ''}, 'diff': 0}
            ),
            url
        )

    @staticmethod
    def run_test_task(rel_target):
        logging.debug("Running test task")
        findurl_task = findurl.FindUrl.run_batch_task(
            [FindUrlRelease.make_simple_doc("vk", "vk.com"), FindUrlRelease.make_simple_doc("yandex", "yandex.ru")],
            "hamster.yandex.ru",
            "",
            "http://{}.n.yandex-team.ru".format(rel_target.replace('_', '-')),
        )
        findurl.FindUrl.wait_task(findurl_task)
        eh.ensure(
            findurl_task.status in [findurl.FindUrl.Task.State.done_all_in_top10, findurl.FindUrl.Task.State.done],
            "Test task status is not 'done'",
        )

    def nanny_deploy(self, task_id):
        nanny_client = client.NannyClient(
            api_url='http://nanny.yandex-team.ru/',
            oauth_token=sdk2.Vault.data('nanny_oauth_token'),
        )
        rel_targets = (
            (["findurl_priemka"] if self.Parameters.run_priemka else [])
            + list(self.Parameters.release_target)
        )
        rel_targets_success = []
        rel_targets_failure = []
        for rel_target in rel_targets:
            try:
                self.wait_working_tasks(rel_target)
                old_time_value = nanny_client.get_service_current_state(rel_target)["content"]["summary"]["entered"]
                if not FindUrlRelease.is_ya_package_in_nanny(nanny_client, rel_target, task_id):
                    nanny_client.update_service_sandbox_file(
                        rel_target, u'YA_PACKAGE', task_id, deploy=True, recipe="default"
                    )
                    self.nanny_check_deploy(nanny_client, rel_target, task_id, old_time_value)
                self.run_test_task(rel_target)
            except Exception as e:
                eh.log_exception("Failed to deploy on {}.".format(rel_target), e)
                rel_targets_failure.append(rel_target)
                if rel_target == "findurl_priemka":
                    logging.debug("Raising exception...")
                    raise
                else:
                    logging.debug("Continuing...")
                    continue
            rel_targets_success.append(rel_target)
        info = ''
        if rel_targets_success:
            info += 'Deployed package from task {} to service{} {}.'.format(
                task_id,
                's:' if len(rel_targets_success) > 1 else '',
                ', '.join(rel_targets_success),
            )
        if rel_targets_failure:
            info += 'Failed to deploy package from task {} to service{} {}.'.format(
                task_id,
                's:' if len(rel_targets_failure) > 1 else '',
                ', '.join(rel_targets_failure),
            )
        self.set_info(info)

    def on_execute(self):
        if self.Parameters.release_source == "task":
            ya_make_task_id = self.Parameters.ya_make_task_id
        else:
            if not self.Context.subtasks_launched:
                self.launch_and_wait_subtasks()

            children = list(self.find())
            for child in children:
                self.check_subtask_status(child)

            eh.ensure(not self.Context.errors, 'One of child tasks failed unexpectedly')

            ya_make_task_id = children[0].id

        self.nanny_deploy(str(ya_make_task_id))
