# coding: utf-8

import os
import re
import json
import shutil
import logging
import datetime
import traceback

import sandbox.sandboxsdk as ssdk

import sandbox.common.types.client as ctc

import sandbox.projects.common.arcadia.sdk as asdk
from sandbox.projects.common import constants as const
from sandbox.projects.common.build.YaMake import YaMakeTask
from sandbox.projects.common.build import parameters as build_params
from sandbox.projects.common.build.arcadia_project_misc import get_arcadia_project_base_target_params

SIMPLE_FUZZ_CMD = "ya make -A --sanitize=address --sanitize-coverage=trace-div,trace-gep --fuzz-opts='-max_total_time=300' --fuzzing"


class FuzzMaxTotalTime(ssdk.parameters.SandboxIntegerParameter):
    name = 'max_total_time'
    required = False
    description = 'max_total_time for fuzzing'
    default_value = 3600


class EnableCurfew(ssdk.parameters.SandboxBoolParameter):
    name = 'curfew'
    required = False
    description = 'Enable curfew for task (do not run between 10:00 and 23:00)'
    default_value = True


class TokenVaultOwner(ssdk.parameters.SandboxStringParameter):
    name = 'token_vault_owner'
    required = True
    description = 'Vault owner for sandbox token'


class TokenVaultKeyName(ssdk.parameters.SandboxStringParameter):
    name = 'token_vault_key'
    required = True
    description = 'Vault key for sandbox token'


class ForceMinimization(ssdk.parameters.SandboxBoolParameter):
    name = 'force_minimization'
    required = False
    description = 'Always minimize corpus after fuzzing stage'
    default_value = False


class FuzzYaMakeTask(YaMakeTask):

    type = 'FUZZ_YA_MAKE_TASK'

    input_parameters = [
        build_params.ArcadiaUrl,
        build_params.BuildSystem,
        TokenVaultOwner,
        TokenVaultKeyName,
        build_params.VaultOwner,
        build_params.VaultKeyName,
        FuzzMaxTotalTime,
        ForceMinimization,
        EnableCurfew,
    ] + get_arcadia_project_base_target_params().params

    def pre_execute(self):
        if len(self.get_targets()) > 1:
            raise Exception('You can fuzz only one target')

        max_total_time = self.ctx['max_total_time']

        self.ctx[const.BUILD_TYPE_KEY] = const.DEBUG_BUILD_TYPE
        self.ctx[const.TESTS_REQUESTED] = True
        self.ctx[const.RESOURCE_OWNER] = self.owner
        # XXX move to task options
        self.ctx[const.SANITIZE] = 'address'
        self.ctx[const.SANITIZE_COVERAGE] = 'trace-div,trace-gep'
        self.ctx[const.SANDBOX_UPLOADED_RESOURCE_TTL] = '40'  # days
        self.ctx[const.KEEP_ON] = True

        self.ctx[const.FUZZING] = True
        self.ctx[const.FUZZ_OPTS] = "-max_total_time={}".format(max_total_time)
        self.ctx[const.FUZZ_FORCE_MINIMIZATION] = self.ctx.get(ForceMinimization.name, False)
        self.ctx[const.CHECK_RETURN_CODE] = False

        # setup token
        if not self.ctx.get('env_vars'):
            self.ctx['env_vars'] = "YA_TOKEN='$(vault:value:AUTOCHECK:devtools-fuzzing-ya-token)'"

        self.corpuses_dir = self.get_corpuses_dir()
        self.old_corpus = self.get_corpus_json(self.corpuses_dir)

        if self.on_dist():
            self.ctx[const.MAKE_CONTEXT_ON_DISTBUILD] = True
            self.ctx[const.BUILD_EXECUTION_TIME] = 4 * 60 * 60  # 4h
            vault_owner = self.ctx.get('token_vault_owner', self.author)
            vault_key_name = self.ctx.get('token_vault_key')
            if vault_owner and vault_key_name:
                self.ctx[const.SANDBOX_TOKEN] = str(self.get_vault_data(vault_owner, vault_key_name))

        # XXX DEVTOOLSSUPPORT-9498
        if self.is_maps_mobile():
            self.ctx[const.DEFINITION_FLAGS_KEY] = ' '.join(filter(None, [self.ctx.get(const.DEFINITION_FLAGS_KEY), '-DMAPSMOBI_BUILD_TARGET=yes']))

    @staticmethod
    def raise_internal_error(text):
        # internal problems should lead to the INTERNAL_ERROR task status to notify maintainers only (DEVTOOLS-4221)
        raise Exception('Internal errors:\n{}'.format(text))

    @staticmethod
    def raise_user_error(text):
        # user problems should lead to the FAILURE task status to notify owners (DEVTOOLS-4221)
        # TODO raise ssdk.errors.SandboxTaskFailureError
        raise Exception("Fuzzing has encountered problems - test owners should take a look.\n{}\nContact devtools@yandex-team.ru if you are stuck".format(text.strip("\n")))

    def on_dist(self):
        return self.ctx[const.BUILD_SYSTEM_KEY] in (const.SEMI_DISTBUILD_BUILD_SYSTEM, const.DISTBUILD_BUILD_SYSTEM, const.DISTBUILD_FORCE_BUILD_SYSTEM)

    def is_maps_mobile(self):
        target = self.get_targets()[0]
        return target.startswith('maps/mobile')

    def on_enqueue(self):
        self.client_tags = ctc.Tag.MULTISLOT if self.on_dist() else ctc.Tag.HDD
        self.ctx[const.SANDBOX_TAGS] = str(self.client_tags)  # SDK1 YaMakeTask expects tags here

        YaMakeTask.on_enqueue(self)

    def do_execute(self):
        if self.on_dist() and self.ctx.get('curfew', True) and (datetime.datetime.now().hour >= 10 or datetime.datetime.now().hour < 1):
            raise Exception("Do not run fuzzing on distbuild in rush hours")

        try:
            YaMakeTask.do_execute(self)
        except Exception:
            self.raise_internal_error("YaMakeTask.do_execute failed with error:\n{}".format(traceback.format_exc()))
        self.do_post_build()
        logging.debug("do_execute finished")

    def get_corpuses_dir(self):
        parsed_url = ssdk.svn.Arcadia.parse_url(self.ctx[const.ARCADIA_URL_KEY])
        url = "arcadia:{}/fuzzing".format(asdk._get_svn_url(parsed_url))
        corpuses_dir = ssdk.paths.make_folder('corpuses_dir')
        ssdk.svn.Arcadia.checkout(url, corpuses_dir, revision=parsed_url.revision)
        return corpuses_dir

    def get_corpus_json(self, corupus_dir):
        project_path = self.get_targets()[0]
        filename = os.path.join(corupus_dir, project_path, "corpus.json")
        if not os.path.exists(filename):
            return {}
        with open(filename) as afile:
            try:
                return json.load(afile)
            except ValueError:
                self.raise_internal_error("Failed to parse corpus data {}: {}\n".format(filename, traceback.format_exc()))
        return {}

    def do_post_build(self):
        logging.debug("Post build stage")
        output_dir = self.get_output_dir()
        self.check_results(output_dir)
        self.set_task_info_description(self.load_metrics(output_dir))

        res_filename = os.path.join(output_dir, "fuzz_result_node.json")
        logging.info("Result file with changes: %s", res_filename)
        if not os.path.exists(res_filename):
            self.raise_internal_error("Result file with changes doesn't exist: {}".format(res_filename))

        if self.ctx.get('build_returncode') not in [0, 10]:
            self.raise_internal_error("ya make failed with {} returncode".format(self.ctx.get('build_returncode')))

        with open(res_filename) as afile:
            data = json.load(afile)
        if data:
            self._commit(output_dir, data[0])
        else:
            logging.info("No new data was mined")

    def check_results(self, output_dir):
        project_path = self.get_targets()[0]
        logs_dir = ssdk.paths.get_logs_folder()

        if self.on_dist():
            db_res_filename = os.path.join(logs_dir, "distbuild-logs/distbuild-result.json")
            if not os.path.exists(db_res_filename):
                self.raise_internal_error("File doesn't exist: {}".format(db_res_filename))
                return

            with open(db_res_filename) as afile:
                db_res = json.load(afile)
            reasons = db_res.get("failed_results", {}).get("reasons", {})
            for uid, node in reasons.iteritems():
                # report node may fail - it's ok, fuzzing found something
                if uid.startswith("report"):
                    continue
                elif uid.startswith("fuzz_minimize-"):
                    self.raise_user_error("Minimization failed:\n{}".format(json.dumps({uid: node}, indent=4, sort_keys=True)))
                else:
                    self.raise_internal_error("Found failed results:\n{}".format(json.dumps({uid: node}, indent=4, sort_keys=True)))

        res_filename = os.path.join(output_dir, "results.json")
        if not os.path.exists(res_filename):
            self.raise_internal_error("File doesn't exist: {}".format(res_filename))
            return
        with open(res_filename) as afile:
            resjson = json.load(afile)
        if not resjson.get("results"):
            self.raise_user_error(
                "Fuzz test wasn't launched - take a look at output_1.html\n"
                "Try manually run fuzzing: {}\n"
                "".format(SIMPLE_FUZZ_CMD.format(project_path))
            )
            return

        suite = None
        for e in resjson["results"]:
            if e.get("suite"):
                suite = e
                break
        if suite.get('error_type') and 'minimization takes to much time' in suite['error_type']:
            self.raise_user_error(self._strip_markup(suite['rich-snippet']))
            return

    def _commit(self, output_dir, entry):
        resource_id = entry["resource_id"]
        project_path = entry["project_path"]

        with open(os.path.join(output_dir, "fuzzing/{}/corpus.json".format(project_path))) as afile:
            mined_corpus = json.load(afile)

        merged_resources = set(self.old_corpus.get("corpus_parts", [])) - set(mined_corpus["corpus_parts"])
        logging.info("Merged resources: %s", str(sorted(merged_resources)))

        commit_msg = "Updated corpus with new mined data by sandbox task #{} for target {}".format(self.id, project_path)
        if merged_resources:
            commit_msg += " Merged {} resources.".format(len(merged_resources))
        commit_msg += "\nSKIP_CHECK"

        logging.info("Going to make commit with message: %s", commit_msg)

        attempts = 10
        for _ in range(attempts):
            local_corpus = os.path.join(self.corpuses_dir, project_path, "corpus.json")
            corpus = {}
            if os.path.exists(local_corpus):
                with open(local_corpus) as afile:
                    corpus = json.load(afile)

            if "corpus_parts" not in corpus:
                corpus["corpus_parts"] = []

            # new corpus data might appear while we were fuzzing in the repository,
            # and this data wasn't merged - don't lose it
            corpus["corpus_parts"] = list(set(corpus["corpus_parts"]) - merged_resources)
            assert resource_id not in corpus["corpus_parts"], (resource_id, corpus)
            corpus["corpus_parts"].append(resource_id)

            new_target_path = local_corpus
            while new_target_path:
                parent = os.path.dirname(new_target_path)
                if os.path.exists(parent):
                    break
                new_target_path = parent

            logging.info("Target file: %s (corpus: %s)", new_target_path, local_corpus)
            logging.info("Final target corpus data: %s", json.dumps(corpus, indent=4, sort_keys=True))

            corpus_dir = os.path.dirname(local_corpus)
            if not os.path.exists(corpus_dir):
                os.makedirs(corpus_dir)

            with open(local_corpus, "w") as afile:
                json.dump(corpus, afile, indent=4, sort_keys=True)

            # this might be new file
            try:
                ssdk.svn.Svn.add(local_corpus, parents=True)
            except ssdk.errors.SandboxSvnError:
                # already versioned
                pass

            try:
                ssdk.svn.Arcadia.commit(new_target_path, commit_msg, 'zomb-sandbox-rw')
            except ssdk.errors.SandboxSvnError:
                if os.path.isfile(new_target_path):
                    os.unlink(new_target_path)
                else:
                    shutil.rmtree(new_target_path)
                ssdk.svn.Svn.revert(new_target_path, recursive=True)
                ssdk.svn.Svn.update(new_target_path)
                continue
            else:
                return
        raise Exception("Failed to commit changes in {} attempts".format(attempts))

    def set_task_info_description(self, metrics):
        msg = ""
        for target in sorted(metrics):
            msg += "Target: {}\n".format(target)
            msg += "{}\n".format(json.dumps(metrics[target], sort_keys=True, indent=4))
        self.set_info(msg)

    def load_metrics(self, output_dir):
        project_path = self.get_targets()[0]
        tracefile = os.path.join(output_dir, project_path, "test-results/fuzz/ytest.report.trace")
        if not os.path.exists(tracefile):
            self.raise_internal_error("Tracefile doesn't exist: {}".format(tracefile))
            return {}

        with open(tracefile) as afile:
            lines = afile.readlines()
        test_status_line = lines[-1]
        try:
            data = json.loads(test_status_line)
        except ValueError as e:
            self.raise_internal_error("Tracefile decode error: {}".format(e))
            return {}

        metrics = {}
        try:
            metrics[project_path] = data["value"]["metrics"]
        except KeyError as e:
            self.raise_internal_error("Failed to find metrics: {}".format(e))
        return metrics

    def _strip_markup(self, text):
        return re.sub(r"\[\[[a-z0-9]+]]", r"", text)


__Task__ = FuzzYaMakeTask
