# coding: utf-8
import json
import logging

import requests
import yaml

from sandbox import common
from sandbox.projects.common import file_utils
from sandbox.projects.resource_types import AbstractResource
from sandbox.projects import rtc_balancer
from sandbox.sandboxsdk.parameters import SandboxBoolParameter, SandboxIntegerParameter, SandboxStringParameter
import sandbox.sandboxsdk.process as sdk_process
import sandbox.sandboxsdk.svn as sdk_svn
import sandbox.sandboxsdk.task as sdk_task


class BALANACER_SLB_CHECKS_YAML(AbstractResource):
    """YAML with active checks of slb"""
    auto_backup = False
    executable = False
    releasable = True
    releasers = rtc_balancer.cplb_releasers
    release_subscribers = ['cplb-releases']
    share = False


class DebugMode(SandboxBoolParameter):
    """For suspending task"""

    name = 'debug_mode'
    description = 'Enable debug mode'
    default_value = False


class CommitRevision(SandboxStringParameter):
    name = 'commit_revision'
    description = 'Commit revision'
    default_value = ''


DEFAULT_SKYLI_PERCENT = 0


class SkyliPercent(SandboxIntegerParameter):
    name = 'skyli_percent'
    description = 'SKYLI percent'
    default_value = DEFAULT_SKYLI_PERCENT


class GencfgGroupNotFound(Exception):
    pass


class Task(sdk_task.SandboxTask):
    """Generate balancer active checks yaml file for ansible juggler"""

    type = "GENERATE_BALANCER_ACTIVE_CHECKS"
    input_parameters = [DebugMode, CommitRevision, SkyliPercent]

    def fill_scheme(self, host='', header_host='', service='', nanny_service='', path='',
                    check_instances_via_https=False, tags=[]):
        return {
            'juggler_check': '',
            'args': {
                'host': host,
                'service': service,
                'namespace': 'search.balancer',
                'children': 'NANNY%{}:{}:all'.format(nanny_service, service),
                'aggregator': 'timed_more_than_limit_is_problem',
                'aggregator_kwargs': {
                    'ignore_nodata': 1,
                    'limits': [
                        {
                            'time_start': 0,
                            'time_end': 23,
                            'day_start': 1,
                            'day_end': 7,
                            'warn': '30%',
                            'crit': '50%',
                        }
                    ]
                },
                'active': 'https' if check_instances_via_https else 'http',
                'active_kwargs': {
                    'ok_codes': 200,
                    'path': path,
                    'headers': {
                        'Host': header_host,
                    },
                },
                'tags': tags,
            }
        }

    @classmethod
    def suggest_geo_from_nanny_service_id(cls, service_id):
        known_locations = ('sas', 'vla', 'man', 'msk', 'myt', 'iva')
        geo = service_id.split('_')[-1].lower()
        if geo in known_locations:
            return geo
        if service_id.endswith('-yp'):
            # service_id most likely is smth like rtc-balancer-mlp-yandex-net-iva-yp
            parts = service_id[:-3].split('-')
            if len(parts) > 1:
                geo = parts[-1].lower()
                if geo in known_locations:
                    return geo
        return 'all'

    def on_execute(self):
        self.awacs_oauth_token = self.get_vault_data('CPLB', 'awacs-oauth-token')
        self.nanny_oauth_token = self.get_vault_data('CPLB', 'nanny-oauth-token')
        self.session = requests.Session()
        self.session.headers['Content-Type'] = 'application/json'

        juggler_checks_root = sdk_svn.Arcadia.get_arcadia_src_dir(
            'arcadia:/arc/trunk/arcadia/juggler/playbooks/search/'
        )
        monitoring_root = sdk_svn.Arcadia.get_arcadia_src_dir(
            'arcadia:/arc/trunk/arcadia/infra/awacs/templates/aux-balancers-data/@{}'.format(
                self.ctx['commit_revision']
            )
        )
        with open('{}/data.json'.format(monitoring_root)) as ifile:
            try:
                monitoring_data = json.load(ifile)
            except ValueError as err:
                raise common.errors.TaskFailure('Loading of data.json file ended with error: {}'.format(err))

        if '__awacs_defined_fqdns__' in monitoring_data:
            del monitoring_data['__awacs_defined_fqdns__']

        service_checks = False
        try:
            result = list()

            for fqdn in monitoring_data:
                if monitoring_data[fqdn].get('skip_probe'):
                    continue

                if not monitoring_data[fqdn].get('url'):
                    logging.warn('{}: "url" option is missed'.format(fqdn))
                    continue

                header_host = monitoring_data[fqdn].get('host') or fqdn
                for nanny_service in monitoring_data[fqdn].get('services'):
                    geo = self.suggest_geo_from_nanny_service_id(nanny_service)

                    result.append(self.fill_scheme(
                        host=fqdn,
                        header_host=header_host,
                        service=geo,
                        nanny_service=nanny_service,
                        path=monitoring_data[fqdn]['url'],
                        check_instances_via_https=monitoring_data[fqdn].get('check_instances_via_https'),
                        tags=['{}.balancer.services'.format(geo)]
                    ))

            file_utils.write_file(
                'active_checks.yaml',
                yaml.safe_dump(result, default_flow_style=False, explicit_start=True)
            )
            service_checks = True

        except IOError as err:
            logging.fatal("Error to open active_checks.yaml file: " % err)

        # CPLB-513. Simple monitoring of slb
        slb_checks = False
        try:
            result = list()
            for key, value in monitoring_data.iteritems():
                if value.get('skip_probe'):
                    continue

                if not monitoring_data[fqdn].get('url'):
                    logging.warn('{}: "url" option is missed'.format(fqdn))
                    continue

                if value.get('https_only'):
                    port = 443
                elif value.get('port'):
                    port = value['port']
                else:
                    port = 80

                result.append({
                    'juggler_check': '',
                    'args': {
                        'host': key,
                        'service': 'slb-check',
                        'namespace': 'search.balancer',
                        'active': 'https' if value.get('https_only') else 'http',
                        'active_kwargs': {
                            'port': port,
                            'ok_codes': 200,
                            'use_all_dc': True,
                            'aggr_mode': 'worst',
                            'timeout': 4,
                            'path': value['url'],
                            'headers': {
                                'Host': value.get('host', key)
                            }
                        },
                        'tags': ['cplb', key],
                        'notifications': [
                            {
                                'template': 'on_status_change',
                                'status': ['CRIT'],
                                'method': ['telegram'],
                                'login': ['slb_checks_notifications'],
                                'fallback_delay': 0
                            }
                        ]
                    }
                })

            file_utils.write_file(
                'slb_checks.yaml',
                yaml.safe_dump(
                    result,
                    default_flow_style=False,
                    explicit_start=True
                )
            )
            slb_checks = True

        except IOError as err:
            logging.fatal("Error to open slb_checks.yaml file: " % err)

        if self.ctx['debug_mode']:
            self.suspend()

        if service_checks:
            self.create_resource(
                arch='any',
                description=self.descr,
                resource_path='active_checks.yaml',
                resource_type='BALANACER_ACTIVE_CHECKS_YAML',
            )

            sdk_process.run_process(
                [
                    'cp',
                    'active_checks.yaml',
                    '{}/roles/balancer.services/tasks/main.yml'.format(
                        juggler_checks_root
                    )
                ],
                log_prefix='copy_active_checks'
            )

            sdk_svn.Arcadia.commit(
                [
                    '{}/roles/balancer.services/tasks/main.yml'.format(juggler_checks_root)
                ],
                'Auto update active checks of balancer services\nSKIP_CHECK',
                'zomb-sandbox-rw'
            )

        if slb_checks:
            self.create_resource(
                arch='any',
                description=self.descr,
                resource_path='slb_checks.yaml',
                resource_type=BALANACER_SLB_CHECKS_YAML,
            )

            sdk_process.run_process(
                [
                    'cp',
                    'slb_checks.yaml',
                    '{}/roles/balancer.slbs/tasks/main.yml'.format(
                        juggler_checks_root
                    )
                ],
                log_prefix='copy_slb_checks'
            )

            sdk_svn.Arcadia.commit(
                [
                    '{}/roles/balancer.slbs/tasks/main.yml'.format(juggler_checks_root)
                ],
                'Auto update active checks of slb\nSKIP_CHECK',
                'zomb-sandbox-rw'
            )


__Task__ = Task
