import sys
import time
import logging


from core.utils import get_ch_formatted_date_from_timestamp, floor_timestamp_to_period
from core.query import run_query, run_single_int_query, get_roughly_last_event_date_from_table


SUFFIXES_AND_PERIODS = {
    "2m": 60 * 2,
    "15m": 60 * 15,
    "1h": 60 * 60,
    "1d": 60 * 60 * 24,
}

INSTANCEUSAGE_TABLE = 'instanceusage'
HOSTUSAGE_TABLE = 'hostusage'
QLOUDUSAGE_TABLE = 'qloudusage'
OPENSTACKUSAGE_TABLE = 'openstackusage'
ABC_TABLE = 'abcusage'
METAPRJ_TABLE = 'metaprjusage'
YPUSAGE_TABLE = 'ypusage'


class HostToPowerUnits(object):
    """
    Implements logic for working with host-to-power-units mapping

    |-------------------------.......-------------->  time
       ^        ^       ^                 ^       ^
       |        |       |                 |       |
       |        ts      |     .......     |      now
      mapping1       mapping2            mappingN

    In this example, for ts: mapping1 is in the past, mapping2..N is in the future (relative to ts),
    so mapping1 is used for re-calculating from instance_cpuusage to instance_cpuusage_power_units

    So this class reads and wraps mapping for a specific timestamp ts

    """

    def __init__(self, ts):
        self.current_ts = self.calculate_current_ts(ts)

    @staticmethod
    def get_next_ts(ts):
        """
        Gets next mapping (in the relative future) for timestamp ts
        """
        end_ed = get_ch_formatted_date_from_timestamp(ts + 3 * 24 * 60 * 60)
        start_ed = get_ch_formatted_date_from_timestamp(ts - 3 * 24 * 60 * 60)

        query = "SELECT min(ts) FROM host_resources WHERE ts > {}  AND eventDate <= '{}' and eventDate >= '{}'".format(ts, end_ed, start_ed)
        return run_single_int_query(query)

    @staticmethod
    def get_last_ts():
        """
        Gets the most recent timestamp available in the clickhouse table
        """
        query = "SELECT max(ts) FROM host_resources"
        return run_single_int_query(query)

    def calculate_current_ts(self, ts):
        end_ed = get_ch_formatted_date_from_timestamp(ts + 7 * 24 * 60 * 60)
        start_ed = get_ch_formatted_date_from_timestamp(ts - 7 * 24 * 60 * 60)

        query = "SELECT max(ts) FROM host_resources WHERE ts <= {} AND eventDate <= '{}' and eventDate >= '{}'".format(ts, end_ed, start_ed)
        current_ts = run_single_int_query(query)
        assert current_ts is None or current_ts

        return current_ts


def get_last_ts_for_aggregated_table(table_name, suffix):
    aggregated_table_name = '{}_aggregated_{}'.format(table_name, suffix)
    last_event_date = get_roughly_last_event_date_from_table(aggregated_table_name, days_to_watch=5 * 365)

    if last_event_date:
        query = "SELECT max(ts) FROM {aggregated_table_name} WHERE eventDate >= '{ed}'".format(
            aggregated_table_name=aggregated_table_name,
            ed=last_event_date,
        )
        query_result = run_query(query, retries=9, retries_timeout_seconds=0.5, request_timeout_seconds=900)
        return int(query_result[0][0]) if len(query_result) > 0 else 0
    return 0


def is_aggregated_table_empty(table_name, suffix):
    query = "SELECT count() FROM (select * from {table_name}_aggregated_{suffix} limit 1);".format(
        table_name=table_name,
        suffix=suffix,
    )
    query_result = run_query(query, retries=9, retries_timeout_seconds=0.5, request_timeout_seconds=900)
    return not query_result


def is_period_empty(table_name, start_ts, end_ts):
    query = """
        SELECT count() FROM (
            SELECT *
            FROM {table_name}
            WHERE eventDate >= '{ed1}' AND eventDate <= '{ed2}' AND ts >= {ts1} AND ts < {ts2}
            LIMIT 1
        )
    """.format(
        table_name=table_name,
        ed1=get_ch_formatted_date_from_timestamp(start_ts),
        ed2=get_ch_formatted_date_from_timestamp(end_ts),
        ts1=start_ts,
        ts2=end_ts,
    )
    query_result = run_query(query, retries=9, retries_timeout_seconds=0.5, request_timeout_seconds=900)
    return not query_result


def get_first_ts_for_table(table_name):
    query = "SELECT min(ts) FROM {table_name};".format(
        table_name=table_name,
    )
    query_result = run_query(query, retries=9, retries_timeout_seconds=0.5, request_timeout_seconds=900)
    return int(query_result[0][0]) if len(query_result) > 0 else 0


def _validate_query(query):
    cnt = 0
    for table in (HOSTUSAGE_TABLE, INSTANCEUSAGE_TABLE, QLOUDUSAGE_TABLE, OPENSTACKUSAGE_TABLE, ABC_TABLE, METAPRJ_TABLE, YPUSAGE_TABLE):
        if table in query:
            cnt += 1
    assert cnt == 1


def _generate_zoomed_tables(query_template, start_ts, end_ts, suffix, period):
    """
    Does join on host table between host_resources (host to power units) table
    """
    assert start_ts % period == 0
    assert end_ts % period == 0

    host_to_units = HostToPowerUnits(start_ts)
    host_resources_ts = host_to_units.current_ts

    if host_to_units.current_ts is None:
        return

    query = query_template.format(
        suffix=suffix,
        period=period,
        ts1=start_ts,
        ts2=end_ts,
        ed1=get_ch_formatted_date_from_timestamp(start_ts),
        ed2=get_ch_formatted_date_from_timestamp(end_ts),
        host_resources_ts=host_resources_ts,
        host_resources_ed=get_ch_formatted_date_from_timestamp(host_resources_ts)
    )

    _validate_query(query)
    run_query(query, retries=9, retries_timeout_seconds=0.5)


def generate_zoomed_instanceusage_tables(start_ts, end_ts, suffix, period):
    query_template = """
        INSERT INTO instanceusage_aggregated_{suffix}
        SELECT host, port, ts - ts % {period} as ts_period, group, eventDate,
              major_tag, minor_tag, version,
              toFloat32(avg(instance_memusage)),
              toFloat32(avg(instance_cpuusage)),
              toFloat32(avg(instance_cpuusage_power_units)),
              toFloat32(avg(instance_anon_memusage)),
              toFloat32(avg(instance_net_rx)),
              toFloat32(avg(instance_net_tx))
        FROM (SELECT *, instance_cpuusage*cpu_power as instance_cpuusage_power_units
              FROM (
                    SELECT * FROM instanceusage
                    WHERE eventDate >= '{ed1}' AND eventDate <= '{ed2}' AND ts >= {ts1} AND ts < {ts2})
                    ALL LEFT OUTER JOIN (
                        select host, cpu_power
                        from host_resources
                        where eventDate='{host_resources_ed}' AND ts={host_resources_ts}
                    ) USING host
             )
        GROUP BY group, host, port, eventDate, ts_period, major_tag, minor_tag, version;
    """
    _generate_zoomed_tables(query_template, start_ts, end_ts, suffix, period)


def generate_zoomed_hostusage_tables(start_ts, end_ts, suffix, period):
    query_template = """
        INSERT INTO hostusage_aggregated_{suffix}
        SELECT host, ts - ts % {period} as ts_period, eventDate,
              toFloat32(avg(mem_usage)),
              toFloat32(avg(cpu_usage)),
              toFloat32(avg(cpu_usage_power_units)),
              toFloat32(avg(hdd_total)),
              toFloat32(avg(hdd_usage)),
              toFloat32(avg(ssd_total)),
              toFloat32(avg(ssd_usage)),
              toFloat32(avg(net_rx)),
              toFloat32(avg(net_tx))
        FROM (SELECT *, cpu_usage*cpu_power as cpu_usage_power_units
              FROM (
                    SELECT * FROM hostusage
                    WHERE eventDate >= '{ed1}' AND eventDate <= '{ed2}' AND ts >= {ts1} AND ts < {ts2})
                    ALL LEFT OUTER JOIN (
                        select host, cpu_power
                        from host_resources
                        where eventDate='{host_resources_ed}' AND ts={host_resources_ts}
                    ) USING host
             )
        GROUP BY host, eventDate, ts_period;
    """
    _generate_zoomed_tables(query_template, start_ts, end_ts, suffix, period)


def generate_zoomed_qloudusage_tables(start_ts, end_ts, suffix, period):
    query_template = """
        INSERT INTO qloudusage_aggregated_{suffix}
        SELECT host, instanceId, ts - ts % {period} as ts_period, eventDate,
               service, environ, app, project, installType,
               toFloat32(avg(mem_usage)),
               toFloat32(avg(cpu_usage)),
               toFloat32(avg(cpu_usage_power_units)),
               toFloat32(avg(cpu_usage_cores)),
               toFloat32(avg(net_rx)),
               toFloat32(avg(net_tx))
        FROM (SELECT *, cpu_usage*cpu_power as cpu_usage_power_units,
                     cpu_usage*cpu_cores as cpu_usage_cores
              FROM (
                    SELECT * FROM qloudusage
                    WHERE eventDate >= '{ed1}' AND eventDate <= '{ed2}' AND ts >= {ts1} AND ts < {ts2})
                    ALL LEFT OUTER JOIN (
                        select host, cpu_power, cpu_cores
                        from host_resources
                        where eventDate='{host_resources_ed}' AND ts={host_resources_ts}
                    ) USING host
             )
        GROUP BY host, instanceId, eventDate, ts_period, service, environ, app, project, installType;
    """
    _generate_zoomed_tables(query_template, start_ts, end_ts, suffix, period)


def generate_zoomed_openstackusage_tables(start_ts, end_ts, suffix, period):
    query_template = """
        INSERT INTO openstackusage_aggregated_{suffix}
        SELECT host, instanceId, ts - ts % {period} as ts_period, eventDate, projectId,
               toFloat32(avg(mem_usage)),
               toFloat32(avg(cpu_usage)),
               toFloat32(avg(cpu_usage_power_units)),
               toFloat32(avg(cpu_usage_cores))
        FROM (SELECT *, cpu_usage*cpu_power as cpu_usage_power_units,
                     cpu_usage*cpu_cores as cpu_usage_cores
              FROM (
                    SELECT * FROM openstackusage
                    WHERE eventDate >= '{ed1}' AND eventDate <= '{ed2}' AND ts >= {ts1} AND ts < {ts2})
                    ALL LEFT OUTER JOIN (
                        select host, cpu_power, cpu_cores
                        from host_resources
                        where eventDate='{host_resources_ed}' AND ts={host_resources_ts}
                    ) USING host
             )
        GROUP BY host, instanceId, eventDate, ts_period, projectId;
    """
    _generate_zoomed_tables(query_template, start_ts, end_ts, suffix, period)


def generate_zoomed_abc_tables(start_ts, end_ts, suffix, period):
    query_template = """
        INSERT INTO abcusage_aggregated_{suffix}
        SELECT ts - ts % {period} as ts_period, eventDate, l1project, l2project, l3project, l4project, l5project, l6project, l7project, l8project, l9project, l10project,
               toFloat32(avg(memory_usage)),
               toFloat32(avg(memory_total)),
               toFloat32(avg(cpu_usage)),
               toFloat32(avg(cpu_total)),
               toFloat32(avg(hdd_usage)),
               toFloat32(avg(hdd_total)),
               toFloat32(avg(ssd_usage)),
               toFloat32(avg(ssd_total)),
               toInt32(avg(hosts_with_stats)),
               toInt32(avg(hosts_total))
        FROM (SELECT *
              FROM (
                    SELECT * FROM abcusage
                    WHERE eventDate >= '{ed1}' AND eventDate <= '{ed2}' AND ts >= {ts1} AND ts < {ts2})
             )
        GROUP BY eventDate, ts_period, l1project, l2project, l3project, l4project, l5project, l6project, l7project, l8project, l9project, l10project;
    """
    _generate_zoomed_tables(query_template, start_ts, end_ts, suffix, period)


def generate_zoomed_metaprj_tables(start_ts, end_ts, suffix, period):
    query_template = """
        INSERT INTO metaprjusage_aggregated_{suffix}
        SELECT ts - ts % {period} as ts_period, eventDate, metaprj,
               toFloat32(avg(cpu_usage)),
               toFloat32(avg(cpu_allocated)),
               toFloat32(avg(memory_usage)),
               toFloat32(avg(memory_allocated))
        FROM (SELECT *
              FROM (
                    SELECT * FROM metaprjusage
                    WHERE eventDate >= '{ed1}' AND eventDate <= '{ed2}' AND ts >= {ts1} AND ts < {ts2})
             )
        GROUP BY eventDate, ts_period, metaprj;
    """
    _generate_zoomed_tables(query_template, start_ts, end_ts, suffix, period)


def generate_zoomed_ypusage_tables(start_ts, end_ts, suffix, period):
    query_template = """
        INSERT INTO ypusage_aggregated_{suffix}
        SELECT host, port, ts - ts % {period} as ts_period, group, eventDate,
              toFloat32(avg(instance_memusage)),
              toFloat32(avg(instance_memallocated)),
              toFloat32(avg(instance_cpuusage)),
              toFloat32(avg(instance_cpuusage_power_units)),
              toFloat32(avg(instance_cpuallocated_power_units)),
              toFloat32(avg(instance_anon_memusage)),
              toFloat32(avg(instance_net_rx)),
              toFloat32(avg(instance_net_tx))
        FROM (SELECT *, instance_cpuusage*cpu_power as instance_cpuusage_power_units, instance_cpuallocated*cpu_power as instance_cpuallocated_power_units
              FROM (
                    SELECT * FROM ypusage
                    WHERE eventDate >= '{ed1}' AND eventDate <= '{ed2}' AND ts >= {ts1} AND ts < {ts2})
                    ALL LEFT OUTER JOIN (
                        select host, cpu_power
                        from host_resources
                        where eventDate='{host_resources_ed}' AND ts={host_resources_ts}
                    ) USING host
             )
        GROUP BY group, host, port, eventDate, ts_period;
    """
    _generate_zoomed_tables(query_template, start_ts, end_ts, suffix, period)


def _generate_day_zoomed_table_if_necessary(table_name, query_template):
    # TODO: the approach here (i.e. INSERT SELECT can be used for generate_zoomed_tables too
    # TODO: also, add weekly tables (?)

    period = 24 * 60 * 60
    logging.info("Trying to generate data for table %s: suffix = 1d and period = %s", table_name, period)

    start_ts = get_last_ts_for_aggregated_table(table_name, "1d") + period

    end_ts = get_last_ts_for_aggregated_table(table_name, "1h")
    end_ts = (end_ts + 3 * 60 * 60) - (end_ts + 3 * 60 * 60) % (24 * 60 * 60) - 3 * 60 * 60

    ed1 = get_ch_formatted_date_from_timestamp(start_ts)
    ed2 = get_ch_formatted_date_from_timestamp(end_ts - period)

    if start_ts >= end_ts:
        logging.info("Skipping generating data for table %s: suffix = 1d and period = %s", table_name, period)
        return False

    query = query_template.format(tablename=table_name, suffix="1d", ed1=ed1, ed2=ed2)
    _validate_query(query)

    logging.info('Run query {}'.format(query))

    run_query(query, retries=9, retries_timeout_seconds=0.5)
    return True


def generate_day_zoomed_instanceusage_table_if_necessary():
    # We have to do this stuff with 3*60*60 because we are in Moscow time
    # (so calculating timestamp of the start of the day is somewhat tricky)
    query_template = """
        INSERT INTO {tablename}_aggregated_{suffix}
        SELECT host,
               port,
               (((min(ts) + 3*60*60) - (min(ts) + 3*60*60) % (24*3600)) - 3*60*60) as ts,
               group,
               eventDate,
               major_tag,
               minor_tag,
               version,
               toFloat32(sum(instance_memusage)/24) as instance_memusage,
               toFloat32(sum(instance_cpuusage)/24) as instance_cpuusage,
               toFloat32(sum(instance_cpuusage_power_units)/24) as instance_cpuusage_power_units,
               toFloat32(sum(instance_anon_memusage)/24) as instance_anon_memusage,
               toFloat32(sum(instance_net_rx)/24) as instance_net_rx,
               toFloat32(sum(instance_net_tx)/24) as instance_net_tx
        FROM {tablename}_aggregated_1h WHERE eventDate >= '{ed1}' AND eventDate <= '{ed2}'
        GROUP by host, port, group, eventDate, major_tag, minor_tag, version
    """
    return _generate_day_zoomed_table_if_necessary(INSTANCEUSAGE_TABLE, query_template)


def generate_day_zoomed_hostusage_table_if_necessary():
    query_template = """
        INSERT INTO {tablename}_aggregated_{suffix}
        SELECT host,
               (((min(ts) + 3*60*60) - (min(ts) + 3*60*60) % (24*3600)) - 3*60*60) as ts,
               eventDate,
               toFloat32(sum(mem_usage)/24) as mem_usage,
               toFloat32(sum(cpu_usage)/24) as cpu_usage,
               toFloat32(sum(cpu_usage_power_units)/24) as cpu_usage_power_units,
               toFloat32(sum(hdd_total)/24) as hdd_total,
               toFloat32(sum(hdd_usage)/24) as hdd_usage,
               toFloat32(sum(ssd_total)/24) as ssd_total,
               toFloat32(sum(ssd_usage)/24) as ssd_usage,
               toFloat32(sum(net_rx)/24) as net_rx,
               toFloat32(sum(net_tx)/24) as net_tx
        FROM {tablename}_aggregated_1h WHERE eventDate >= '{ed1}' AND eventDate <= '{ed2}'
        GROUP by host, eventDate
    """
    return _generate_day_zoomed_table_if_necessary(HOSTUSAGE_TABLE, query_template)


def generate_day_zoomed_qloudusage_table_if_necessary():
    query_template = """
        INSERT INTO {tablename}_aggregated_{suffix}
        SELECT host,
               instanceId,
               (((min(ts) + 3*60*60) - (min(ts) + 3*60*60) % (24*3600)) - 3*60*60) as ts,
               eventDate,
               service, environ, app, project, installType,
               toFloat32(sum(mem_usage)/24) as mem_usage,
               toFloat32(sum(cpu_usage)/24) as cpu_usage,
               toFloat32(sum(cpu_usage_power_units)/24) as cpu_usage_power_units,
               toFloat32(sum(cpu_usage_cores)/24) as cpu_usage_cores,
               toFloat32(sum(net_rx)/24) as net_rx,
               toFloat32(sum(net_tx)/24) as net_tx
        FROM {tablename}_aggregated_1h WHERE eventDate >= '{ed1}' AND eventDate <= '{ed2}'
        GROUP BY host, instanceId, eventDate, service, environ, app, project, installType;
    """
    return _generate_day_zoomed_table_if_necessary(QLOUDUSAGE_TABLE, query_template)


def generate_day_zoomed_openstackusage_table_if_necessary():
    query_template = """
        INSERT INTO {tablename}_aggregated_{suffix}
        SELECT host,
               instanceId,
               (((min(ts) + 3*60*60) - (min(ts) + 3*60*60) % (24*3600)) - 3*60*60) as ts,
               eventDate,
               projectId,
               toFloat32(sum(mem_usage)/24) as mem_usage,
               toFloat32(sum(cpu_usage)/24) as cpu_usage,
               toFloat32(sum(cpu_usage_power_units)/24) as cpu_usage_power_units,
               toFloat32(sum(cpu_usage_cores)/24) as cpu_usage_cores
        FROM {tablename}_aggregated_1h WHERE eventDate >= '{ed1}' AND eventDate <= '{ed2}'
        GROUP BY host, instanceId, eventDate, projectId;
    """
    return _generate_day_zoomed_table_if_necessary(OPENSTACKUSAGE_TABLE, query_template)


def generate_day_zoomed_abc_table_if_necessary():
    query_template = """
        INSERT INTO {tablename}_aggregated_{suffix}
        SELECT (((min(ts) + 3*60*60) - (min(ts) + 3*60*60) % (24*3600)) - 3*60*60) as ts,
               eventDate,
               l1project,
               l2project,
               l3project,
               l4project,
               l5project,
               l6project,
               l7project,
               l8project,
               l9project,
               l10project,
               toFloat32(sum(memory_usage)/24) as memory_usage,
               toFloat32(sum(memory_total)/24) as memory_total,
               toFloat32(sum(cpu_usage)/24) as cpu_usage,
               toFloat32(sum(cpu_total)/24) as cpu_total,
               toFloat32(sum(hdd_usage)/24) as hdd_usage,
               toFloat32(sum(hdd_total)/24) as hdd_total,
               toFloat32(sum(ssd_usage)/24) as ssd_usage,
               toFloat32(sum(ssd_total)/24) as ssd_total,
               toInt32(sum(hosts_with_stats)/24) as hosts_with_stats,
               toInt32(sum(hosts_total)/24) as hosts_total
        FROM {tablename}_aggregated_1h WHERE eventDate >= '{ed1}' AND eventDate <= '{ed2}'
        GROUP BY eventDate, l1project, l2project, l3project, l4project, l5project, l6project, l7project, l8project, l9project, l10project;
    """
    return _generate_day_zoomed_table_if_necessary(ABC_TABLE, query_template)


def generate_day_zoomed_metaprj_table_if_necessary():
    query_template = """
        INSERT INTO {tablename}_aggregated_{suffix}
        SELECT (((min(ts) + 3*60*60) - (min(ts) + 3*60*60) % (24*3600)) - 3*60*60) as ts,
               eventDate,
               metaprj,
               toFloat32(sum(cpu_usage)/24) as cpu_usage,
               toFloat32(sum(cpu_allocated)/24) as cpu_allocated,
               toFloat32(sum(memory_usage)/24) as memory_usage,
               toFloat32(sum(memory_allocated)/24) as memory_allocated
        FROM {tablename}_aggregated_1h WHERE eventDate >= '{ed1}' AND eventDate <= '{ed2}'
        GROUP BY eventDate, metaprj;
    """
    return _generate_day_zoomed_table_if_necessary(METAPRJ_TABLE, query_template)


def generate_day_zoomed_ypusage_table_if_necessary():
    # We have to do this stuff with 3*60*60 because we are in Moscow time
    # (so calculating timestamp of the start of the day is somewhat tricky)
    query_template = """
        INSERT INTO {tablename}_aggregated_{suffix}
        SELECT host,
               port,
               (((min(ts) + 3*60*60) - (min(ts) + 3*60*60) % (24*3600)) - 3*60*60) as ts,
               group,
               eventDate,
               toFloat32(sum(instance_memusage)/24) as instance_memusage,
               toFloat32(sum(instance_memallocated)/24) as instance_memallocated,
               toFloat32(sum(instance_cpuusage)/24) as instance_cpuusage,
               toFloat32(sum(instance_cpuusage_power_units)/24) as instance_cpuusage_power_units,
               toFloat32(sum(instance_cpuallocated_power_units)/24) as instance_cpuallocated_power_units,
               toFloat32(sum(instance_anon_memusage)/24) as instance_anon_memusage,
               toFloat32(sum(instance_net_rx)/24) as instance_net_rx,
               toFloat32(sum(instance_net_tx)/24) as instance_net_tx
        FROM {tablename}_aggregated_1h WHERE eventDate >= '{ed1}' AND eventDate <= '{ed2}'
        GROUP by host, port, group, eventDate
    """
    return _generate_day_zoomed_table_if_necessary(YPUSAGE_TABLE, query_template)


def _get_start_timestamp(table_name, suffix):
    period = SUFFIXES_AND_PERIODS[suffix]

    if is_aggregated_table_empty(table_name, suffix):
        logging.warn('aggregated table %s is empty!', table_name)
        start_ts = get_first_ts_for_table(table_name)
        start_ts = floor_timestamp_to_period(start_ts, period)
        logging.warn('starting from %s', start_ts)
    else:
        start_ts = get_last_ts_for_aggregated_table(table_name, suffix)

    return start_ts + period


def _get_end_timestamp(suffix, start_ts):
    # we wait for ~10 minutes until all the data is in sync/uploaded
    wait_period = 10 * 60
    max_compute_period = SUFFIXES_AND_PERIODS['1h']
    period = SUFFIXES_AND_PERIODS[suffix]

    assert period <= max_compute_period

    possible_end_timestamps = [start_ts + max_compute_period, int(time.time()) - wait_period]
    h2p_next_ts = HostToPowerUnits.get_next_ts(start_ts)
    logging.info("Host_to_units.get_next_ts(%s) = %s", start_ts, h2p_next_ts)
    # We should definitely stop before next host-to-power-mapping becomes available
    # Otherwise computed cpu usage power units field might be incorrect
    if h2p_next_ts:
        possible_end_timestamps.append(h2p_next_ts + period)

    end_ts = min(possible_end_timestamps)
    logging.debug("possible_end_timestamps = %s; min = %s", possible_end_timestamps, end_ts)
    end_ts = floor_timestamp_to_period(end_ts, period)
    logging.debug("computed end_ts = %s", end_ts)

    return end_ts


def _get_start_end_timestamps(table_name, suffix):
    period = SUFFIXES_AND_PERIODS[suffix]

    start_ts = _get_start_timestamp(table_name, suffix)
    end_ts = _get_end_timestamp(suffix, start_ts)

    while (
            start_ts < end_ts
            and is_period_empty(table_name, start_ts, end_ts)
    ):
        logging.warn('period [%s, %s] is empty, skipping it', start_ts, end_ts)
        start_ts += period
        end_ts = _get_end_timestamp(suffix, start_ts)

    return start_ts, end_ts


def _generate_short_zoomed_tables_if_necessary(table_name, suffix):
    assert table_name in (HOSTUSAGE_TABLE, INSTANCEUSAGE_TABLE, QLOUDUSAGE_TABLE, OPENSTACKUSAGE_TABLE, ABC_TABLE, METAPRJ_TABLE, YPUSAGE_TABLE)

    period = SUFFIXES_AND_PERIODS[suffix]
    logging.info("Trying to generate data for table: suffix = %s and period = %s", suffix, period)

    start_ts, end_ts = _get_start_end_timestamps(table_name, suffix)

    logging.info("Auto-compute: start_ts = %s and end_ts = %s. Range len = %s", start_ts, end_ts, end_ts - start_ts)
    if start_ts >= end_ts:
        logging.info("Skipping generating data for table %s: suffix = %s and period = %s", table_name, suffix, period)
        return False
    elif table_name == INSTANCEUSAGE_TABLE:
        generate_zoomed_instanceusage_tables(start_ts, end_ts, suffix, period)
    elif table_name == HOSTUSAGE_TABLE:
        generate_zoomed_hostusage_tables(start_ts, end_ts, suffix, period)
    elif table_name == QLOUDUSAGE_TABLE:
        generate_zoomed_qloudusage_tables(start_ts, end_ts, suffix, period)
    elif table_name == OPENSTACKUSAGE_TABLE:
        generate_zoomed_openstackusage_tables(start_ts, end_ts, suffix, period)
    elif table_name == ABC_TABLE:
        generate_zoomed_abc_tables(start_ts, end_ts, suffix, period)
    elif table_name == METAPRJ_TABLE:
        generate_zoomed_metaprj_tables(start_ts, end_ts, suffix, period)
    elif table_name == YPUSAGE_TABLE:
        generate_zoomed_ypusage_tables(start_ts, end_ts, suffix, period)
    return True


def generate_short_zoomed_instanceusage_tables_if_necessary(suffix):
    return _generate_short_zoomed_tables_if_necessary(INSTANCEUSAGE_TABLE, suffix)


def generate_short_zoomed_hostusage_tables_if_necessary(suffix):
    return _generate_short_zoomed_tables_if_necessary(HOSTUSAGE_TABLE, suffix)


def generate_short_zoomed_qloudusage_tables_if_necessary(suffix):
    return _generate_short_zoomed_tables_if_necessary(QLOUDUSAGE_TABLE, suffix)


def generate_short_zoomed_openstackusage_tables_if_necessary(suffix):
    return _generate_short_zoomed_tables_if_necessary(OPENSTACKUSAGE_TABLE, suffix)


def generate_short_zoomed_abc_tables_if_necessary(suffix):
    return _generate_short_zoomed_tables_if_necessary(ABC_TABLE, suffix)


def generate_short_zoomed_metaprj_tables_if_necessary(suffix):
    return _generate_short_zoomed_tables_if_necessary(METAPRJ_TABLE, suffix)


def generate_short_zoomed_ypusage_tables_if_necessary(suffix):
    return _generate_short_zoomed_tables_if_necessary(YPUSAGE_TABLE, suffix)


def setup_logging():
    root_logger = logging.getLogger()

    root_logger.setLevel(logging.DEBUG)
    logging.getLogger('requests.packages.urllib3.connectionpool').setLevel(logging.WARNING)

    handler = logging.StreamHandler()

    fmt = "%(asctime)s:%(levelname)s:%(name)s:%(message)s"
    formatter = logging.Formatter(fmt=fmt)
    handler.setFormatter(formatter)

    root_logger.addHandler(handler)


def regular_update_process(suffixes=tuple(SUFFIXES_AND_PERIODS), sleep_func=time.sleep):
    logging.info("Starting regular update process for tables %s", suffixes)

    while True:
        skipped_suffixes = 0
        for suffix in suffixes:
            if suffix != "1d":
                was_generated = any([
                    generate_short_zoomed_hostusage_tables_if_necessary(suffix),
                    generate_short_zoomed_instanceusage_tables_if_necessary(suffix),
                    generate_short_zoomed_qloudusage_tables_if_necessary(suffix),
                    generate_short_zoomed_openstackusage_tables_if_necessary(suffix),
                    generate_short_zoomed_abc_tables_if_necessary(suffix),
                    # generate_short_zoomed_metaprj_tables_if_necessary(suffix),
                    generate_short_zoomed_ypusage_tables_if_necessary(suffix),
                ])
            else:
                was_generated = any([
                    generate_day_zoomed_hostusage_table_if_necessary(),
                    generate_day_zoomed_instanceusage_table_if_necessary(),
                    generate_day_zoomed_qloudusage_table_if_necessary(),
                    generate_day_zoomed_openstackusage_table_if_necessary(),
                    generate_day_zoomed_abc_table_if_necessary(),
                    # generate_day_zoomed_metaprj_table_if_necessary(),
                    generate_day_zoomed_ypusage_table_if_necessary(),
                ])

            if not was_generated:
                skipped_suffixes += 1

        if skipped_suffixes == len(suffixes):
            logging.info("Skipped all tables, sleeping 5 minutes")
            sleep_func(5 * 60)


if __name__ == "__main__":
    setup_logging()
    assert len(sys.argv) < 3
    if len(sys.argv) == 1:
        regular_update_process()
    elif len(sys.argv) == 2:
        assert sys.argv[1] in SUFFIXES_AND_PERIODS
        regular_update_process([sys.argv[1]])
    else:
        assert False
