# coding: utf-8

import errno
import json
import os
import subprocess
import time
import urllib2
from collections import defaultdict
from copy import deepcopy
from datetime import datetime
from logging import info, error

import sandbox.projects.common.yabs.graphite as graphite
from sandbox.common.utils import server_url
from sandbox.projects import resource_types
from sandbox.projects.GenerateLinearModelBinaryDump import (
    LinearModelDump,
    Location,
    LinearModelId,
    TruncateOptions,
    CorrectionOptions,
    ValidationOptions,
    VwDumpBinary,
    UnpackDumpBinary,
    GenerateLmDumpsBinary,
    GenerateLinearModelBinaryDump,
    GraphitePathPrefix,
    DumpType,
    DumpName,
    DumpCreateTime,
    DumpLastLogDate,
    Ttl,
    Released,
    UpdateReasonId,
    UpdateReasonMessage,
    CompressionLevel,
    Token,
    TokenOwner,
    YtFolder,
    YtProxy,
    ClusterType,
    get_sharding_base_resource_ids,
    get_latest_resource,
    get_meta as get_linear_model_binary_dump_meta
)
from sandbox.sandboxsdk import environments
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.parameters import (
    SandboxIntegerParameter,
    SandboxRadioParameter,
    SandboxBoolParameter,
    SandboxStringParameter,
    ListRepeater,
    LastReleasedResource
)
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.task import SandboxTask


class ShardsNumber(SandboxIntegerParameter):
    name = 'total_shards'
    description = 'Number of shards'
    required = True


class SabtasksKillTimeout(SandboxIntegerParameter):
    name = 'kill_timeout'
    description = 'Timeout to kill subtasks'
    default_value = 60 * 60 * 4


class ForceUpdate(SandboxBoolParameter):
    name = 'force_update'
    description = 'Force update all models'
    default_value = False


class ReleaseStatus(SandboxRadioParameter):
    choices = [(_, _) for _ in ("none", "prestable", "stable", "testing", "unstable")]
    description = 'release status'
    default_value = "none"
    name = 'release_status'


class GlobalCompressionLevel(SandboxStringParameter):
    name = 'global_compression_level'
    description = 'Compression level for all output dumps'
    required = False
    default_value = None


class MysqlHosts(ListRepeater, SandboxStringParameter):
    name = 'mysql_hosts'
    description = 'mysql hosts with LinearModelConfig table'
    default_value = ["bsbackup02i.yandex.ru", "bsbackup02g.yandex.ru"]


SUPPORTED_COMPRESSION_LEVELS = {"2", "None", "3"}
SUPPORTED_CLUSTER_TYPES = {'yabs', 'bs', 'all', "None"}


def get_psql_params(group_name='Postgresql'):

    class PostgresqlParameters(object):
        class PsqlBinary(LastReleasedResource):
            name = 'psql_binary'
            description = 'Postgresql binary'
            resource_type = 'PSQL_BINARY'
            group = group_name

        class Host(SandboxStringParameter):
            name = 'psql_host'
            description = 'Postgresql server host name'
            group = group_name

        class Port(SandboxIntegerParameter):
            name = 'psql_port'
            description = 'Postgresql port'
            group = group_name

        class Database(SandboxStringParameter):
            name = 'psql_database'
            description = 'Database name'
            group = group_name

        class User(SandboxStringParameter):
            name = 'psql_user'
            description = 'User name'
            group = group_name

        class PasswordID(SandboxStringParameter):
            name = 'psql_password_id'
            description = 'Vault password id'
            group = group_name

        class Retries(SandboxIntegerParameter):
            name = 'psql_retries'
            description = 'Retries to connect and send query'
            group = group_name

        params = (
            PsqlBinary,
            Host,
            Port,
            Database,
            User,
            PasswordID,
            Retries
        )

    return PostgresqlParameters


class StatusEnum(object):
    FORCE_UPDATE = 0
    NEW_MODEL = 1
    NEED_RESHARDING_NO_SHARDED_YET = 2
    NEED_RESHARDING_UPDATED_BASES = 3
    UPDATED_MODEL = 4
    NO_DUMP = 5
    ONE_SHOT = 6
    CONSISTENT_MODEL = 7
    NOT_CONSISTENT_STAT_SHARDS_STATE = 8


class Status(object):
    def __init__(self, status, message, need_update):
        self.status = status
        self.message = message
        self.need_update = need_update


def get_meta(task):

    bin_dumps_meta = []

    for s in channel.sandbox.list_tasks(task_type="GENERATE_LINEAR_MODEL_BINARY_DUMP", parent_id=task.id, limit=500):
        if s.is_finished():
            bin_dumps_meta.append(get_linear_model_binary_dump_meta(s))

    on_release_resources = channel.sandbox.list_resources(
        resource_type=resource_types.LM_DUMPS_LIST,
        task_id=task.id,
        limit=100
    )

    releases = {}

    for resource in on_release_resources:
        location = resource.attributes['location']
        shard = int(resource.attributes.get('shard', 1))
        releases[(location, shard)] = {
            "release_resource_id": resource.id,
            "release_resource_created": resource.timestamp
        }

    launcher_released = None

    for dump in bin_dumps_meta:
        release_created = None
        for stat in dump['shards_stats']:
            release = releases.get((dump['location'], stat['shard']), None)
            if release is None:
                release = {'release_resource_id': None, 'release_resource_created': None}
            stat.update(release)
            release_created = max(release_created, stat['release_resource_created'])
        dump['shards_stats'] = json.dumps(dump['shards_stats'])
        dump['release_created'] = datetime.fromtimestamp(release_created).isoformat(' ')
        launcher_released = max(launcher_released, release_created)

    for dump in bin_dumps_meta:
        dump.update({
            "launcher_id": task.id,
            "launcher_created": datetime.fromtimestamp(task.timestamp).isoformat(' '),
            "launcher_started": datetime.fromtimestamp(task.timestamp_start).isoformat(' '),
            "launcher_release_type": task.ctx[ReleaseStatus.name].encode('utf-8'),
            "launcher_released": datetime.fromtimestamp(launcher_released).isoformat(' ')
        })

    return bin_dumps_meta


def generate_query_from_meta(meta):

    if not meta:
        return None

    columns = [
        "launcher_id",
        "launcher_created",
        "launcher_started",
        "launcher_release_type",
        "launcher_released",
        "task_id",
        "task_created",
        "task_started",
        "task_finished",
        "lm_id",
        "location",
        "learn_task_name",
        "last_log_date",
        "dump_txt_type",
        "dump_txt_resource_id",
        "dump_txt_resource_created",
        "shards_num",
        "status",
        "reason_id",
        "reason_message",
        "shards_stats",
        "release_created"
    ]

    lines = []
    for m in meta:
        elements = []

        for h in columns:
            if isinstance(m[h], basestring):
                element = "'%s'" % (m[h].replace('\'', '"'))
            else:
                element = "%s" % m[h]
            elements.append(element)

        lines.append("(%s)" % ', '.join(elements))

    values = ",\n".join(lines)

    query = """
        INSERT INTO GenerateLinearModelBinaryDumpLauncher
        ({})
        VALUES
        {};
        """.format(
        ", ".join(columns),
        values
    )

    return query


def validate_compression_level(compression_level):
    if compression_level not in SUPPORTED_COMPRESSION_LEVELS:
        raise ValueError("Bad compression level: {compression_level}".format(compression_level=compression_level))


def validate_cluster_type(cluster_type):
    if cluster_type not in SUPPORTED_CLUSTER_TYPES:
        raise ValueError("Bad cluster type: {cluster_type}".format(cluster_type=cluster_type))


class GenerateLinearModelBinaryDumpLauncher(SandboxTask):
    type = 'GENERATE_LINEAR_MODEL_BINARY_DUMP_LAUNCHER'

    cores = 1
    required_ram = 4 * 1024
    execution_space = 4 * 1024

    environment = (
        environments.PipEnvironment('MySQL-python', '1.2.5', use_wheel=True),
    )

    input_parameters = (
        ShardsNumber,
        VwDumpBinary,
        UnpackDumpBinary,
        GenerateLmDumpsBinary,
        SabtasksKillTimeout,
        ReleaseStatus,
        ForceUpdate,
        MysqlHosts,
        GlobalCompressionLevel, GraphitePathPrefix, YtFolder, YtProxy, Token, TokenOwner) + get_psql_params().params

    @property
    def psql(self):
        return self.sync_resource(self.ctx['psql_binary'])

    def insert_stat(self, query):
        env = {
            "PSQL_PASSWORD": self.get_vault_data("ML-ENGINE", self.ctx['psql_password_id']),
        }
        p = run_process(
            [
                self.psql,
                '--host', self.ctx['psql_host'],
                '--port', str(self.ctx['psql_port']),
                '--db', self.ctx['psql_database'],
                '--user', self.ctx['psql_user'],
                '--retries', str(self.ctx['psql_retries'])
            ],
            stdin=subprocess.PIPE,
            wait=False,
            outs_to_pipe=True,
            environment=env
        )
        stdout, stderr = p.communicate(input=query)
        if stdout:
            info(stdout)
        if stderr:
            error(stderr)
        if p.returncode:
            raise Exception("Saving metrics exited with code: {:d}".format(p.returncode))

    def connect_to_mysql(self):
        from MySQLdb import connect
        mysql_passwd = self.get_vault_data("bsbackup_mysql_password")

        for i in xrange(30):
            for host in self.ctx[MysqlHosts.name]:
                try:
                    return connect(
                        host=host,
                        port=3306,
                        user="sandbox",
                        db="yabsdb",
                        connect_timeout=10,
                        passwd=mysql_passwd
                    )
                except Exception:
                    error("Can't connect to %s", host, exc_info=True)
                    time.sleep(5)
        raise Exception("Failed to make mysql connection")

    def load_linear_models(self):
        from MySQLdb.cursors import DictCursor

        con = self.connect_to_mysql()
        cur = DictCursor(con)
        cur.execute("""
            SELECT
                LinearModelID,
                Active,
                DumpName,
                DumpType,
                Locations,
                ModelName,
                TruncateOptions,
                ValidatorOptions,
                CorrectionOptions,
                Owners,
                CompressionType,
                ClusterType
            FROM
                LinearModelConfig
            WHERE
                Active in (1,2) and Locations in ('stat', 'meta')
        """)
        models = []
        for model in cur.fetchall():
            locations = model.pop("Locations").strip().split(",")
            if len(locations) == 0:
                error("Model #%d has no Locations")
            for loc in locations:
                new_model = deepcopy(model)
                new_model["Location"] = loc
                models.append(new_model)

        cur.close()
        con.close()

        return models

    def add_release_attr(self, attrs):
        if self.ctx[ReleaseStatus.name] != "none":
            attrs["released"] = str(self.ctx[ReleaseStatus.name])
        return attrs

    def get_text_dump_resource_for_model(self, model):
        if model['DumpType'] == 'online':
            kwargs = dict(
                resource_type=resource_types.ONLINE_LEARNING_DUMP_TXT,
                all_attrs={
                    "ml_task_id": model["DumpName"],
                }
            )
        else:
            kwargs = dict(
                resource_type=resource_types.ML_ENGINE_DUMP,
                all_attrs={
                    "task_id": model["DumpName"],
                }
            )

        if model["Active"] == 2:
            binary_dump_resource = get_latest_resource(
                resource_type=resource_types.LINEAR_MODEL_BINARY_DUMP,
                all_attrs=self.add_release_attr({
                    "linear_model_id": str(model["LinearModelID"])
                })
            )
            if binary_dump_resource is not None:
                last_log_date = binary_dump_resource.attributes.get("last_log_date")
                kwargs["all_attrs"]["last_log_date"] = last_log_date
                info("Oneshot model #%s with last_log_date %s", model["LinearModelID"], last_log_date)
            else:
                info("No last_log_date for oneshot model #%s, update from last text model", model["LinearModelID"])

        return get_latest_resource(sort_by="last_log_date", **kwargs)

    def check_shards_consistency(self, lm_id, last_log_date, task_id):
        shards_number = self.ctx[ShardsNumber.name]

        shards_to_check = set(range(1, shards_number + 1))

        resources = channel.sandbox.list_resources(
            resource_type=resource_types.LINEAR_MODEL_BINARY_DUMP,
            all_attrs=self.add_release_attr({"linear_model_id": str(lm_id), "last_log_date": last_log_date}),
            order_by="-id",
            limit=shards_number,
            task_id=task_id,
            status="READY")

        for res in resources:
            shard = int(res.attributes.get("shard", 0))
            shards_to_check.remove(shard)

        if len(shards_to_check) > 0:
            error("Shards %s are absent for model #%s last_log_date %s", shards_to_check, lm_id, last_log_date)
            return False

        return True

    def check_dump_status(self, model, text_dump_resource, sharding_base_resource_ids, force_update=False):

        lm_id = model["LinearModelID"]

        if text_dump_resource is None:
            message = 'No text dump for linear model id #%s' % lm_id
            return Status(StatusEnum.NO_DUMP, message, False)

        if force_update:
            message = 'Force update of bin dump for linear model id #%s' % lm_id
            return Status(StatusEnum.FORCE_UPDATE, message, True)

        binary_dump_resource = get_latest_resource(
            resource_type=resource_types.LINEAR_MODEL_BINARY_DUMP,
            all_attrs=self.add_release_attr({
                "linear_model_id": str(lm_id)
            })
        )

        if binary_dump_resource is None:
            message = 'New model with id #%s and need generating new bin dump' % lm_id
            return Status(StatusEnum.NEW_MODEL, message, True)

        text_dump_last_log_date = text_dump_resource.attributes["last_log_date"]
        binary_dump_last_log_date = binary_dump_resource.attributes["last_log_date"]
        if model["Location"] == "stat":
            if not self.check_shards_consistency(lm_id, binary_dump_last_log_date, binary_dump_resource.task_id):
                message = "Stat model %s in not consistent state" % lm_id
                return Status(StatusEnum.NOT_CONSISTENT_STAT_SHARDS_STATE, message, True)

            last_update_bases = json.loads(
                binary_dump_resource.attributes.get("sharding_bases", "null")
            )

            if last_update_bases is None:
                message = 'Linear model with id #%s need resharding because not sharded yet' % lm_id
                return Status(StatusEnum.NEED_RESHARDING_NO_SHARDED_YET, message, True)

            for ns in last_update_bases:
                if last_update_bases[ns] != sharding_base_resource_ids[ns]:
                    message = 'Linear model with id #%s need resharding because ' \
                              'sharding bases for namespace "%s" are different: %s(old) vs %s(new)' % \
                              (lm_id, ns, last_update_bases[ns], sharding_base_resource_ids[ns])
                    return Status(StatusEnum.NEED_RESHARDING_UPDATED_BASES, message, True)

        if model["Active"] == 2:
            message = 'Linear model with id #%s is one shot and currently does not need resharding' % lm_id
            return Status(StatusEnum.ONE_SHOT, message, False)

        if text_dump_last_log_date > binary_dump_last_log_date:
            message = 'Updated model with id #%s has last_log_date=%s and need generating of new bin dump' % \
                      (lm_id, text_dump_last_log_date)
            return Status(StatusEnum.UPDATED_MODEL, message, True)
        else:
            message = 'Bin dump for model #%s is already created and does not need resharding' % lm_id
            return Status(StatusEnum.CONSISTENT_MODEL, message, False)

    @staticmethod
    def mkdir_if_not_exist(dirname):
        try:
            os.mkdir(dirname)
        except OSError as e:
            if e.errno != errno.EEXIST:
                raise

    class LMStat(object):

        def __init__(self):
            self.max = 0
            self.count = 0
            self.sum = 0

    def calc_sizes(self, meta_resources, shards_resources):
        by_model = defaultdict(self.LMStat)
        by_location = defaultdict(int)
        for location, dumps in [('meta', meta_resources)] + \
                [('stat_%s' % shard, resources) for shard, resources in shards_resources.iteritems()]:
            for lm_id, dump in dumps:
                by_location[location] += dump.size
                by_model[lm_id].max = max(by_model[lm_id].max, dump.size)
                by_model[lm_id].count += 1
                by_model[lm_id].sum += dump.size

        return {
            'by_location': by_location,
            'by_model': {
                lmid: {
                    'max': stat.max,
                    'avg': stat.sum / float(stat.count)
                } for lmid, stat in by_model.iteritems()
            }
        }

    def set_sizes_info(self, sizes):
        self.set_info(
            'Shards sizes: \n' +
            '\n'.join(['%s: %.3fM' % (location, size / float(1 << 10)) for location, size in sorted(sizes['by_location'].iteritems())])
        )
        self.set_info(
            'Model sizes: \n' +
            '\n'.join([
                '<a href="http://bsmr-server02i.yandex.ru/monitoring/production/vw/models/{lm_id}">{lm_id}</a>: avg={avg:.3g}M, max={max:.3g}M'.format(
                    lm_id=lm_id, avg=stat['avg'] / float(1 << 10), max=stat['max'] / float(1 << 10)
                )
                for lm_id, stat in sorted(sizes['by_model'].iteritems(), key=lambda (a, b): b['avg'], reverse=True)
            ]),
            do_escape=False
        )

    def send_sizes_to_graphite(self, sizes):
        metrics = []
        ts = time.time()
        hostname = urllib2.urlparse.urlparse(server_url()).netloc
        for location, size in sizes['by_location'].iteritems():
            metrics.append(
                graphite.metric_point(
                    graphite.one_hour_metric(
                        self.ctx[ReleaseStatus.name],
                        'lm',
                        'by_location',
                        location,
                        'size',
                        hostname=hostname
                    ),
                    size,
                    ts,
                )
            )

        for lm_id, stat in sizes['by_model'].iteritems():
            for aggr_type, size in stat.iteritems():
                metrics.append(
                    graphite.metric_point(
                        graphite.one_hour_metric(
                            self.ctx[ReleaseStatus.name],
                            'lm',
                            'by_model',
                            str(lm_id),
                            'size_%s' % aggr_type,
                            hostname=hostname
                        ),
                        size,
                        ts,
                    )
                )

        info('send to graphite %s', metrics)

        graphite.Graphite().send(metrics)

    def get_stat_model_shards_resources(self, lm_id, total_shards):

        latest_resource = get_latest_resource(
            resource_type=resource_types.LINEAR_MODEL_BINARY_DUMP,
            all_attrs=self.add_release_attr({
                "linear_model_id": str(lm_id),
                "location": "stat",
                "total_shards": str(total_shards),
            })
        )

        if latest_resource is None:
            error("Stat dump for model #%d not generated!", lm_id)
            return {}

        task_id = latest_resource.task_id
        last_log_date = latest_resource.attributes.get("last_log_date")

        resources = channel.sandbox.list_resources(
            resource_type=resource_types.LINEAR_MODEL_BINARY_DUMP,
            all_attrs=self.add_release_attr({"linear_model_id": str(lm_id), "last_log_date": last_log_date}),
            order_by="-id",
            limit=total_shards,
            task_id=task_id,
            status="READY")

        shards = {int(r.attributes.get("shard")): r for r in resources}

        shards_set = set(shards)
        absent_shards_set = set(range(1, total_shards + 1)) - shards_set

        assert not absent_shards_set, "Model #%s last_log_date=%s doesn't have shards %s" % \
                                      (lm_id, last_log_date, absent_shards_set)

        return shards

    def make_dump_lists_resources(self, models):
        info("Make dump lists resources")
        shards_resources = defaultdict(list)
        meta_resources = []
        total_shards = self.ctx[ShardsNumber.name]

        for m in models:
            lm_id = m["LinearModelID"]

            if m["Location"] == "stat":
                stat_shards = self.get_stat_model_shards_resources(lm_id, total_shards)
                for shard_no, resource in stat_shards.iteritems():
                    info("Resource for model #%s: %s", lm_id, resource)
                    shards_resources[shard_no].append((lm_id, resource,))
            elif m["Location"] == "meta":
                resource = get_latest_resource(
                    resource_type="LINEAR_MODEL_BINARY_DUMP",
                    all_attrs=self.add_release_attr({
                        "linear_model_id": str(lm_id),
                        "location": "meta"
                    })
                )
                if resource is not None:
                    info("Resource for model #%s: %s", lm_id, resource)
                    meta_resources.append((lm_id, resource,))
                else:
                    error("Meta dump for model #%d not generated!", lm_id)

        ###
        #  Write resources
        ###

        resources_lists_dir = "./resources_lists"
        self.mkdir_if_not_exist(resources_lists_dir)

        def write_resources_data(_file, lm_resources):
            _file.write(json.dumps([
                {
                    "LinearModelID": linear_model_id,
                    "ResourceID": linear_model_resource.id,
                    "LastLogDate": linear_model_resource.attributes["last_log_date"],
                    "MD5": linear_model_resource.file_md5,
                } for linear_model_id, linear_model_resource in lm_resources])
            )
            _file.flush()

        sizes = self.calc_sizes(meta_resources, shards_resources)
        self.set_sizes_info(sizes)
        self.send_sizes_to_graphite(sizes)

        for shard_no, stat_resources in shards_resources.iteritems():
            filename = "shard_%s_of_%s_resource_list" % (shard_no, total_shards)
            with open(os.path.join(resources_lists_dir, filename), "w") as shard_file:
                write_resources_data(shard_file, stat_resources)
                self.create_resource(
                    "Resources list for stat shard %d/%d" % (shard_no, total_shards),
                    shard_file.name,
                    "LM_DUMPS_LIST",
                    attributes={
                        "location": "stat",
                        "shard": str(shard_no),
                        "total_shards": str(total_shards),
                    }
                )

        with open(os.path.join(resources_lists_dir, "meta_resource_list"), "w") as meta_file:
            write_resources_data(meta_file, meta_resources)
            self.create_resource(
                "Resources list for meta",
                meta_file.name,
                "LM_DUMPS_LIST",
                attributes={
                    "location": "meta",
                    "total_shards": total_shards
                }
            )

    def self_release(self, release_status):
        return self.create_subtask(
            task_type='RELEASE_ANY',
            description="Release lm resources lists",
            inherit_notifications=True,
            input_parameters={
                'check_already_released': False,
                'release_task_id': self.id,
                'release_status': release_status,
            }
        )

    def launch_task(self, model, text_dump_resource, dump_status):
        shards_number = self.ctx[ShardsNumber.name]
        if model["Location"] == "meta":
            shards_number = 0

        ttl = Ttl.default_value
        if self.ctx[ReleaseStatus.name] == "testing":
            ttl = 'inf'

        info("Launch task for: %s", model)

        compression_level = model['CompressionType'] or self.ctx[GlobalCompressionLevel.name]
        compression_level = str(compression_level)

        # TODO:: remove this check after launch
        validate_compression_level(compression_level)

        cluster_type = model["ClusterType"] or "None"
        validate_cluster_type(cluster_type)

        subtask_ctx = {
            SabtasksKillTimeout.name: self.ctx[SabtasksKillTimeout.name],
            LinearModelDump.name: text_dump_resource.id,
            Location.name: model["Location"],
            ShardsNumber.name: shards_number,
            UpdateReasonId.name: dump_status.status,
            UpdateReasonMessage.name: dump_status.message,
            DumpName.name: model["DumpName"],
            DumpCreateTime.name: text_dump_resource.timestamp,
            DumpLastLogDate.name: text_dump_resource.attributes['last_log_date'],
            LinearModelId.name: model["LinearModelID"],
            TruncateOptions.name: model["TruncateOptions"] or "{}",
            CorrectionOptions.name: model["CorrectionOptions"] or "{}",
            ValidationOptions.name: model["ValidatorOptions"] or "{}",
            GenerateLmDumpsBinary.name: self.ctx[GenerateLmDumpsBinary.name],
            VwDumpBinary.name: self.ctx[VwDumpBinary.name],
            UnpackDumpBinary.name: self.ctx[UnpackDumpBinary.name],
            DumpType.name: model['DumpType'],
            Ttl.name: ttl,
            Released.name: self.ctx[ReleaseStatus.name],
            CompressionLevel.name: compression_level,
            GraphitePathPrefix.name: self.ctx[GraphitePathPrefix.name],
            YtFolder.name: self.ctx[YtFolder.name],
            YtProxy.name: self.ctx[YtProxy.name],
            Token.name: self.ctx[Token.name],
            TokenOwner.name: self.ctx[TokenOwner.name],
            ClusterType.name: cluster_type
        }

        if model["Owners"]:
            notify_users = model["Owners"].strip().split(",")

            notify_launcher_users = self.ctx.get("notify_if_failed")
            if notify_launcher_users:
                notify_users.extend(
                    notify_launcher_users.strip().split(",")
                )

            subtask_ctx.update({
                "notify_via": "email",
                "notify_if_failed": ",".join(filter(None, notify_users))
            })

        return self.create_subtask(
            task_type='GENERATE_LINEAR_MODEL_BINARY_DUMP',
            arch='linux',
            priority=('SERVICE', 'HIGH'),
            input_parameters=subtask_ctx,
            description='Generate dumps for model #%s' % model["LinearModelID"]
        )

    def launch_all_tasks(self, linear_models, force_update=False):
        tasks = []
        base_resource_ids = get_sharding_base_resource_ids(GenerateLinearModelBinaryDump.ns_for_sharding)
        for model in linear_models:
            text_dump_resource = self.get_text_dump_resource_for_model(model)
            dump_status = self.check_dump_status(model, text_dump_resource, base_resource_ids, force_update)
            info(dump_status.message)
            if dump_status.need_update:
                tasks.append(self.launch_task(model, text_dump_resource, dump_status))
        return tasks

    def on_execute(self):
        linear_models = self.load_linear_models()
        if "child_tasks_ids" not in self.ctx:
            info("Linear models: %s", linear_models)
            tasks = self.launch_all_tasks(linear_models, force_update=self.ctx[ForceUpdate.name])
            self.ctx['child_tasks_ids'] = [task.id for task in tasks]
            self.wait_tasks(
                self.ctx['child_tasks_ids'],
                tuple(self.Status.Group.FINISH) + tuple(self.Status.Group.BREAK),
                wait_all=True
            )

        else:
            self.make_dump_lists_resources(linear_models)

            if self.ctx[ReleaseStatus.name] != "none":
                self.self_release(self.ctx[ReleaseStatus.name])

    def on_release(self, release_parameters):
        SandboxTask.on_release(self, release_parameters)
        meta = get_meta(self)

        if meta:
            info('Start saving meta for released dumps')
            query = generate_query_from_meta(meta)
            self.insert_stat(query)
        else:
            info('No any dumps have been released')


__Task__ = GenerateLinearModelBinaryDumpLauncher
