# coding: utf8

import sandbox.common.types.misc as ctm
from sandbox.common.types.client import Tag
from sandbox.projects import resource_types
from sandbox.projects.common.nanny import nanny
from sandbox.sandboxsdk.parameters import SandboxBoolParameter
import logging
import os
import requests
import sandbox.sandboxsdk.process as sdk_process
import sandbox.sandboxsdk.task as sdk_task
import sandbox.sandboxsdk.svn as sdk_svn
import subprocess
import time


class DebugMode(SandboxBoolParameter):
    """For suspending task"""

    name = 'debug_mode'
    description = 'Enable debug mode'
    default_value = False


class Task(nanny.ReleaseToNannyTask, sdk_task.SandboxTask):
    """Generate ocsp response files for search balancers"""

    type = 'GENERATE_OCSP_RESPONSE_FILES'
    execution_space = 512
    required_ram = 1024
    input_parameters = [DebugMode]
    dns = ctm.DnsType.DNS64
    client_tags = Tag.LINUX_PRECISE

    def on_release(self, additional_parameters):
        nanny.ReleaseToNannyTask.on_release(self, additional_parameters)
        self.mark_released_resources('stable', ttl=10)

    def on_execute(self):
        with self.memoize_stage.test:
            certs_dir = sdk_svn.Arcadia.get_arcadia_src_dir(
                'arcadia:/robots/trunk/genconf/resources/balancer/certs'
            )
            current_time = int(time.time())
            os.mkdir(os.path.join(os.path.abspath(''), 'ocsp'))

            if self.ctx['debug_mode']:
                logging.info('[INFO] Dir with certs here: {}'.format(certs_dir))
                self.suspend()

            counter = 0
            for file in os.listdir(certs_dir):
                if file.endswith('.pem') and file.startswith('allCAs-'):
                    ocsp_uri_proc = sdk_process.run_process(
                        [
                            'openssl', 'x509', '-noout', '-ocsp_uri',
                            '-in', '{}/{}'.format(certs_dir, file)
                        ],
                        work_dir=certs_dir,
                        stdout=subprocess.PIPE,
                        log_prefix='ocsp_uri',
                    )
                    ocsp_uri = ocsp_uri_proc.stdout.read().rstrip()

                    if not ocsp_uri:
                        continue

                    issuer_proc = sdk_process.run_process(
                        [
                            'openssl', 'x509', '-noout', '-issuer',
                            '-in', '{}/{}'.format(certs_dir, file)
                        ],
                        work_dir=certs_dir,
                        stdout=subprocess.PIPE,
                        log_prefix='issuer'
                    )
                    issuer = issuer_proc.stdout.read().rstrip()

                    if issuer == 'issuer= /C=FR/O=KEYNECTIS/CN=CLASS 2 KEYNECTIS CA':
                        CA_cert = '{}/root/CA-Certplus.pem'.format(certs_dir)
                        no_nonce = False

                    elif issuer == 'issuer= /C=PL/O=Unizeto Technologies S.A./OU=Certum Certification Authority/CN=Certum Level IV CA':
                        CA_cert = '{}/root/CA-Unizeto.pem'.format(certs_dir)
                        no_nonce = '-no_nonce'

                    elif issuer == 'issuer= /C=RU/O=Yandex LLC/OU=Yandex Certification Authority/CN=Yandex CA':
                        CA_cert = '{}/root/CA-Yandex.pem'.format(certs_dir)
                        no_nonce = '-no_nonce'

                    elif issuer == 'issuer= /C=BE/O=GlobalSign nv-sa/CN=GlobalSign Organization Validation CA - SHA256 - G2':
                        CA_cert = '{}/root/CA-GlobalSign.pem'.format(certs_dir)
                        no_nonce = ''

                    elif issuer == 'issuer= /C=BE/O=GlobalSign nv-sa/CN=GlobalSign Organization Validation CA - G2':
                        CA_cert = '{}/root/CA-GlobalSignSha1.pem'.format(certs_dir)
                        no_nonce = False

                    else:
                        logging.warn('WANRNING: There is no CAfile for {}'.format(file))
                        continue

                    # make ocsp request
                    tmp_req_file = '{}_temporary.req'.format(file.split('.pem')[0])
                    tmp_der_file = '{}_temporary.der'.format(file.split('.pem')[0])

                    generate_request_args = [
                        'openssl', 'ocsp',
                        '-CAfile', CA_cert,
                        '-issuer', CA_cert,
                        '-cert', '{}/{}'.format(certs_dir, file),
                        '-reqout', os.path.join(os.path.abspath(''), tmp_req_file)
                    ]
                    if no_nonce:
                        generate_request_args.append(no_nonce)

                    try:
                        sdk_process.run_process(
                            generate_request_args,
                            work_dir=certs_dir,
                            log_prefix='generate_request'
                        )

                    except Exception as err:
                        logging.critical(
                            '[ERROR] unexpected error generate ocsp request for {}: {}'.format(
                                file, err
                            )
                        )
                        continue

                    session = requests.Session()
                    session.headers['Content-Type'] = 'application/ocsp-request'
                    for attempt in xrange(3):
                        try:
                            with open(os.path.join(os.path.abspath(''), tmp_req_file), 'rb') as temporary_req:
                                responder_result = session.post(
                                    ocsp_uri,
                                    data=temporary_req
                                )

                        except requests.exceptions.RequestException as err:
                            logging.warning(
                                '[ERROR] Getting response from {} failed: {}'.format(
                                    ocsp_uri, err
                                )
                            )
                            time.sleep((2**attempt))

                        else:
                            if responder_result.status_code != 200:
                                time.sleep((2**attempt))
                                continue

                            elif len(responder_result.content) < 1:
                                time.sleep((2**attempt))
                                continue

                            with open(os.path.join(os.path.abspath(''), tmp_der_file), 'wb') as temporary_result:
                                temporary_result.write(responder_result.content)
                                temporary_result.flush()
                            break

                    # check and validate ocsp response
                    ocsp_reponse_check_args = [
                        'openssl', 'ocsp',
                        '-CAfile', CA_cert,
                        '-issuer', CA_cert,
                        '-cert', '{}/{}'.format(certs_dir, file),
                        '-respin', '{}'.format(os.path.join(os.path.abspath(''), tmp_der_file))
                    ]
                    if no_nonce:
                        ocsp_reponse_check_args.append(no_nonce)

                    try:
                        ocsp_response_check = sdk_process.run_process(
                            ocsp_reponse_check_args,
                            work_dir=certs_dir,
                            stdout=subprocess.PIPE,
                            log_prefix='check_response'
                        )

                    except Exception as err:
                        logging.critical(
                            '[ERROR] unexpected error due check ocsp response for {}: {}'.format(
                                file, err
                            )
                        )
                        continue

                    found_error = False
                    for line in ocsp_response_check.stdout:
                        if 'warning' in line.lower() or 'error' in line.lower():
                            found_error = True

                    if found_error:
                        logging.critical(
                            '[ERROR] Validating of response for {} failed: {}'.format(file, line)
                        )
                        continue

                    ocsp_text_proc = sdk_process.run_process(
                        [
                            'openssl', 'ocsp',
                            '-respin', os.path.join(os.path.abspath(''), tmp_der_file),
                            '-noverify', '-text'
                        ],
                        work_dir=certs_dir,
                        stdout=subprocess.PIPE,
                        log_prefix='verify'
                    )

                    for line in ocsp_text_proc.stdout:
                        if 'This Update' in line:
                            date = line.rstrip().split('Update:')[1]
                            this_update = int(time.mktime(time.strptime(date, ' %b %d %H:%M:%S %Y %Z')))
                        if 'Next Update' in line:
                            date = line.rstrip().split('Update:')[1]
                            next_update = int(time.mktime(time.strptime(date, ' %b %d %H:%M:%S %Y %Z')))

                    if this_update < current_time + 90000 < next_update:  # CPLB-439 current_time + 25h
                        os.rename(
                            os.path.join(os.path.abspath(''), tmp_der_file),
                            os.path.join(os.path.abspath(''), 'ocsp', '{}.der'.format(file.split('.pem')[0]))
                        )
                        counter += 1
                    else:
                        logging.critical(
                            '[ERROR] Status times invalid in {}. This Update: {}; Next Update: {}'.format(
                                tmp_der_file,
                                this_update,
                                next_update
                            )
                        )
                        os.remove(os.path.join(os.path.abspath(''), tmp_der_file))
                        continue

            if counter == 0:
                raise Exception('There are no valid ocsp files at all')

            sdk_process.run_process(
                [
                    'tar', '-czf',
                    os.path.join(os.path.abspath(''), 'ocsp_responses.tgz'),
                    'ocsp'
                ],
                work_dir=os.path.abspath(''),
                log_prefix='tar_files'
            )

            ocsp_responses = self.create_resource(
                arch='any',
                attributes={'ttl': 1},
                description=self.descr,
                resource_path='ocsp_responses.tgz',
                resource_type='OCSP_RESPONSE_FILES_TGZ',
            )
            self.mark_resource_ready(ocsp_responses)

            # wait 12h before release
            self.wait_time(43200)

        releaser = self.create_subtask(
            task_type='RELEASE_ANY',
            inherit_notifications=True,
            input_parameters={
                'check_already_released': str(resource_types.OCSP_RESPONSE_FILES_TGZ),
                'release_task_id': self.id,
                'release_status': 'stable'
            },
            description='ocsp_responses.tgz (task id: {}) autorelease'.format(str(self.id))
        )

        self.info = "Subtask {} runned, waiting it's decision about release.\n\n".format(releaser.id)


__Task__ = Task
