# -*- coding: utf-8 -*-

import os

import sandbox.common.types.client as ctc

from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk import parameters as sp

import sandbox.projects.websearch.middlesearch.resources as ms_resources
import sandbox.projects.images.resource_types as images_resource_type
from sandbox.projects.common import dolbilka
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common import string
from sandbox.projects import resource_types
from sandbox.projects.release_machine import rm_notify


OUT_RESOURCE_KEY = 'out_resource_id'

input2output_resource_type = {
    resource_types.PLAIN_TEXT_QUERIES.name: resource_types.BASESEARCH_PLAN,
    ms_resources.WebMiddlesearchPlainTextQueries.name: ms_resources.MiddlesearchPlan,
    resource_types.IMAGES_MIDDLESEARCH_PLAIN_TEXT_REQUESTS.name: resource_types.IMAGES_MIDDLESEARCH_PLAN,
    resource_types.VIDEO_MIDDLESEARCH_PLAIN_TEXT_REQUESTS.name: resource_types.VIDEO_MIDDLESEARCH_PLAN,
    resource_types.THUMB_DAEMON_REQUESTS.name: resource_types.THUMB_DAEMON_PLAN,
    resource_types.IMPROXY_PLAIN_TEXT_REQUESTS.name: resource_types.IMPROXY_PLAN,
    images_resource_type.IMAGES_THUMBS_REQUESTS.name: images_resource_type.IMAGES_THUMBS_PLAN_2,
    resource_types.MARKET_REPORT_PLAIN_TEXT_QUERIES.name: resource_types.MARKET_REPORT_PLAN,
}


class QueriesParameter(sp.LastReleasedResource):
    name = 'queries_resource_id'
    description = 'Source queries'
    resource_type = input2output_resource_type.keys()


class RpsParameter(sp.SandboxIntegerParameter):
    name = 'rps'
    description = 'Requests per second (0 - unspecified)'


class MinimalPlanSize(sp.SandboxIntegerParameter):
    name = 'minimal_plan_size'
    description = 'Minimal plan size (in bytes)'
    default_value = 10 ** 6
    required = True


class Attr(sp.SandboxStringParameter):
    name = 'resource_attrs'
    description = 'Add attributes to final res (ex.: attr1=v1, attr2=v2)'
    default_value = ''


@rm_notify.notify2()
class GeneratePlanFromQueries(SandboxTask):
    """
        Generates basesearch plan from plain text queries by running
        "d-planner -t -plain -l QUERIES_FILE - o PLAN_FILE"
    """

    type = 'GENERATE_PLAN_FROM_QUERIES'
    client_tags = ctc.Tag.LINUX_PRECISE
    input_parameters = [
        QueriesParameter,
        RpsParameter,
        MinimalPlanSize,
        Attr,
    ]

    def on_enqueue(self):
        SandboxTask.on_enqueue(self)
        attr = string.parse_attrs(self.ctx[Attr.name])
        channel.task = self
        self.ctx[OUT_RESOURCE_KEY] = self.create_resource(
            self.descr,
            'plan.bin',
            input2output_resource_type[channel.sandbox.get_resource(self.ctx[QueriesParameter.name]).type],
            arch='any',
            attributes=attr,
        ).id

    def on_execute(self):
        self.sync_resource(self.ctx[QueriesParameter.name])
        queries_resource = channel.sandbox.get_resource(self.ctx[QueriesParameter.name])
        out_resource = channel.sandbox.get_resource(self.ctx[OUT_RESOURCE_KEY])
        rps = self.ctx[RpsParameter.name] if self.ctx[RpsParameter.name] else None
        dolbilka.convert_queries_to_plan(queries_resource.path, out_resource.path, rps=rps)
        plan_size = os.path.getsize(out_resource.path)
        eh.verify(
            plan_size >= self.ctx[MinimalPlanSize.name],
            "Plan is too small ({} < {})".format(plan_size, self.ctx[MinimalPlanSize.name])
        )


__Task__ = GeneratePlanFromQueries
