# -*- coding: utf-8 -*-

import logging
import os

from sandbox.common.types.client import Tag
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk import parameters as sp
from sandbox.sandboxsdk import paths

from sandbox.projects.tank import executor as tank_executor
from sandbox.projects.tank import phout as tank_phout
from sandbox.projects.common.search import queries as search_queries
from sandbox.projects.common import dolbilka
from sandbox.projects.common import apihelpers
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common import file_utils as fu
from sandbox.projects.common import link_builder as lb
from sandbox.projects.common import templates
from sandbox.projects.common import utils
from sandbox.projects.common import utils2
from sandbox.projects import resource_types
from sandbox.projects.tank.load_resources import resources as tank_resources


class PerformanceTask(sp.TaskSelector):
    """
        Task should contain EXECUTOR_STAT_PHOUT resources and
        'dolbilo_plan_resource_id' in ctx
    """
    name = "performance_task"
    description = "Use slow queries from performance task"
    task_type = [
        "TEST_BASESEARCH_PERFORMANCE",
        "TEST_BASESEARCH_PERFORMANCE_TANK",
        "TEST_BASESEARCH_PERFORMANCE_BEST",
    ]


class NumSlow(sp.SandboxIntegerParameter):
    name = "number_of_slow_requests_to_use"
    description = "Number of slow requests to use"
    default_value = 5


class Detailed(sp.SandboxBoolParameter):
    name = 'detailed_info_for_every_shoot'
    description = 'Detailed info'
    default_value = False


OUT_REQ_RES_ID = "slow_requests_res_id"
OUT_PLAN_RES_ID = "slow_plan_res_id"
_STATS_RES_ID = "stats_res_id"


class GenerateSlowPlan(SandboxTask):
    """
        Generates plan from slowest queries
    """
    type = "GENERATE_SLOW_PLAN"
    environment = tank_phout.ENVIRONMENT
    input_parameters = (PerformanceTask, NumSlow, Detailed)
    client_tags = Tag.LINUX_PRECISE
    cores = 1

    def on_enqueue(self):
        SandboxTask.on_enqueue(self)
        self.ctx[_STATS_RES_ID] = self.create_resource(
            'stats of slow queries', 'stats_of_slow_queries',
            resource_types.OTHER_RESOURCE
        ).id
        self.ctx[OUT_REQ_RES_ID] = self.create_resource(
            'Slow requests', 'slow_requests.txt',
            resource_types.PLAIN_TEXT_QUERIES
        ).id
        self.ctx[OUT_PLAN_RES_ID] = self.create_resource(
            'Slow plan', 'slow_plan',
            resource_types.BASESEARCH_PLAN
        ).id
        # this should be the last expression in the method:
        self.wait_tasks(
            [utils.get_or_default(self.ctx, PerformanceTask)],
            list(self.Status.Group.FINISH + self.Status.Group.BREAK),
            wait_all=True
        )

    def on_execute(self):
        paths.make_folder(
            channel.sandbox.get_resource(self.ctx[_STATS_RES_ID]).path,
            delete_content=True,
        )
        import pandas
        pandas.set_option('max_colwidth', 150)
        perf_task_obj = channel.sandbox.get_task(self.ctx[PerformanceTask.name])
        path_to_reqs = self.get_path_to_requests(perf_task_obj)
        response_times = list(self.get_response_times(perf_task_obj))
        if utils.get_or_default(self.ctx, Detailed):
            for res_id, rt in response_times:
                logging.info("Count slow reqids for resource: %s", res_id)
                slow_reqids = rt[:utils.get_or_default(self.ctx, NumSlow)]
                slow_reqs_dict = self.extract_slow_queries(set(slow_reqids.index), path_to_reqs)
                path_to_stats = os.path.join(
                    channel.sandbox.get_resource(self.ctx[_STATS_RES_ID]).path,
                    "stats_for_{}.html".format(res_id)
                )
                self.write_stats(slow_reqs_dict, slow_reqids, path_to_stats)
        total_slow_reqids = self.total_slow_reqids(response_times)
        slow_reqs_dict = self.extract_slow_queries(set(total_slow_reqids.index), path_to_reqs)
        self.write_reqs_to_resource(slow_reqs_dict)
        path_to_stats = os.path.join(
            channel.sandbox.get_resource(self.ctx[_STATS_RES_ID]).path,
            "total_stats.html"
        )
        self.write_stats(slow_reqs_dict, total_slow_reqids, path_to_stats)
        self.set_info(
            "Slow requests: {}".format(utils2.resource_redirect_link(self.ctx[_STATS_RES_ID], "stats")),
            do_escape=False,
        )

    def total_slow_reqids(self, response_times):
        return self._aggregate_response_times(response_times).nlargest(utils.get_or_default(self.ctx, NumSlow))

    def get_response_times(self, perf_task_obj):
        if perf_task_obj.type == "TEST_BASESEARCH_PERFORMANCE_BEST":
            # choose last child in umbrella task
            perf_task_obj = sorted(channel.sandbox.list_tasks(parent_id=perf_task_obj.id), reverse=True)[0]
        logging.info("Use performance task: %s", perf_task_obj.id)
        stats_res_list = sorted(
            apihelpers.list_task_resources(perf_task_obj.id, resource_types.EXECUTOR_STAT_PHOUT) or
            apihelpers.list_task_resources(perf_task_obj.id, tank_resources.YANDEX_TANK_LOGS),
            key=lambda x: x.id
        )
        logging.info("Got %s relevant performance stats", len(stats_res_list) - 1)
        for st in stats_res_list[1:]:  # first shoot is not relevant
            phout = tank_phout.get_phout(self._path_to_phout(st))
            try:
                sorted_phout = phout.sort("interval_real", ascending=False)
            except AttributeError:
                sorted_phout = phout.sort_values("interval_real", ascending=False)
            # first duplicate remains by default
            tag_time_series = sorted_phout.drop_duplicates("tag").set_index("tag").interval_real
            yield (st.id, tag_time_series)

    @staticmethod
    def _aggregate_response_times(response_times):
        data_sum = None
        for _, data in response_times:
            if data_sum is None:
                data_sum = data.copy()
            else:
                data_sum.add(data)
        return data_sum

    def _path_to_phout(self, res):
        if res.type == resource_types.EXECUTOR_STAT_PHOUT:
            return self.sync_resource(res)
        elif res.type == tank_resources.YANDEX_TANK_LOGS:
            return tank_executor.DolbiloPlugin.get_phout_dump_path(os.path.join(self.sync_resource(res), "artifacts"))
        eh.check_failed("Unknown resource_type = {}".format(res.type))

    def get_path_to_requests(self, task_obj):
        plan_res = channel.sandbox.get_resource(task_obj.ctx["dolbilo_plan_resource_id"])

        if "plan_to_queries_task_id" not in self.ctx:
            self.ctx["plan_to_queries_task_id"] = self.create_subtask(
                task_type="GET_QUERIES_FROM_PLAN",
                description="plan to queries, {}".format(self.descr),
                input_parameters={
                    "plan_resource_id": plan_res.id,
                    "notify_via": "",
                },
            ).id
            utils.wait_all_subtasks_stop()
            return

        res_id = channel.sandbox.get_task(self.ctx["plan_to_queries_task_id"]).ctx["out_resource_id"]
        reqs_path = self.sync_resource(res_id)
        logging.info("Path to queries: %s", reqs_path)
        return reqs_path

    @staticmethod
    def extract_slow_queries(slow_reqids, reqs_path):
        slow_user_reqs = {}
        for line in open(reqs_path):
            for reqid in slow_reqids:
                if reqid.strip("#") in line:  # SEARCH-1812
                    slow_user_reqs[reqid] = line.strip()
                    break
        return slow_user_reqs

    def write_reqs_to_resource(self, slow_full_reqs):
        queries_path = channel.sandbox.get_resource(self.ctx[OUT_REQ_RES_ID]).path
        plan_path = channel.sandbox.get_resource(self.ctx[OUT_PLAN_RES_ID]).path
        with open(queries_path, "w") as out_file:
            out_file.write("\n".join(slow_full_reqs.itervalues()))
        dolbilka.convert_queries_to_plan(queries_path, plan_path)

    def write_stats(self, reqs_dict, slow_reqids, path_to_stats):
        html_stats = templates.get_html_template("simple_table.html")
        rows = []
        heads = ["Time (usec)", "ReqId", "Text", "CGI", "Switches"]
        table_header = self._fill_stats_table_row(heads, "<th>{}</th>")
        row_template = self._fill_stats_table_row(["{}"] * 5, "<td>{}</td>")
        for ind, interval_real in slow_reqids.iteritems():
            cgi = reqs_dict[ind]
            parsed_cgi = search_queries.parse_cgi_params(cgi)
            pron_params = parsed_cgi.get('pron', [])
            user_request = parsed_cgi.get('user_request', [''])[0]
            prons_html = '<br/>\n'.join(pron_params)
            rows.append(row_template.format(
                utils.splitthousands(interval_real).replace(' ', '&nbsp;'),
                ind[1:],
                user_request,
                lb.HREF_TO_ITEM.format(
                    link="http://localhost:7300/yandsearch/" + cgi,
                    name="cgi",
                ),
                prons_html,
            ))
        fu.write_file(
            path_to_stats,
            html_stats % dict(
                title="Slow requests stats",
                table_header=table_header,
                table_rows="\n".join(rows),
            )
        )

    @staticmethod
    def _fill_stats_table_row(row_list, wrapper):
        wrapped_list = (wrapper.format(i) for i in row_list)
        return "<tr>{}</tr>".format("".join(wrapped_list))


__Task__ = GenerateSlowPlan
