# coding: utf-8

# import logging
import os
import tempfile

from sandbox.projects import resource_types
from sandbox.projects.common.nanny import nanny
from sandbox.sandboxsdk import environments
from sandbox.sandboxsdk import process
from sandbox.sandboxsdk.parameters import SandboxIntegerParameter
from sandbox.sandboxsdk.task import SandboxTask


class MaxDeviceIdsParameter(SandboxIntegerParameter):
    name = 'max_device_ids'
    description = 'Maximal number of device_ids, that can be generated'
    required = False
    default_value = 10000000


class GenerateSportDeviceIds(nanny.ReleaseToNannyTask, SandboxTask):
    """
        Генерирует файл search_app_dids.data, в котором находятся did`ы пользователей поискового приложения.
        Затем обновляет ресурс сервиса parallel_biathlon.
        Всё это необходимо для генерации мобильных пушей
    """

    type = 'GENERATE_SPORT_DEVICE_IDS'

    input_parameters = (
        MaxDeviceIdsParameter,
    )

    environment = (
        environments.PipEnvironment('yandex-yt', '0.7.34-0'),
        environments.PipEnvironment('yandex-yt-yson-bindings-skynet')
    )

    YT_PROXY = 'hahn.yt.yandex.net'
    YT_RUNNER = 'yt_generate_device_ids.py'

    def on_execute(self):
        with tempfile.NamedTemporaryFile(bufsize=0) as token_file:
            token_file.write(self.get_vault_data('ustas', 'ustas_hahn_token'))  # TODO: use robot_freshness_hahn_token

            # run YT process in different python
            process.run_process(
                cmd=[
                    '/skynet/python/bin/python',
                    os.path.join(os.path.dirname(__file__), self.YT_RUNNER),
                    '--server=' + self.YT_PROXY,
                    '--token=@' + token_file.name,
                    '--output-file=search_app_dids.data'
                ],
                log_prefix=self.YT_RUNNER,
                wait=True,
                check=True
            )

            search_app_dids_data = self.abs_path('search_app_dids.data')
            with open(search_app_dids_data) as f:
                deviceIdsCount = int(f.readline().strip())
                if deviceIdsCount > self.ctx['max_device_ids']:
                    raise Exception('Too many device_ids was generated: %d. They may eat too much memory in sport_proxy binary' % deviceIdsCount)

            result_resource = self.create_resource(
                description='search_app device_ids for sport wizard',
                resource_path=search_app_dids_data,
                resource_type=resource_types.SPORT_WIZARD_DEVICE_IDS_DATA
            )
            self.mark_resource_ready(result_resource)

            releaser = self.create_subtask(
                task_type='RELEASE_ANY',
                inherit_notifications=True,
                input_parameters={
                    'check_already_released': str(resource_types.SPORT_WIZARD_DEVICE_IDS_DATA),
                    'release_task_id': self.id,
                    'release_status': 'stable',
                },
                description="search_app_dids.data (task id: " + str(self.id) + ') autorelease'
            )
            self.info = "Subtask {} runned, waiting it's decision about release.\n\n".format(releaser.id)

    def on_release(self, additional_parameters):
        nanny.ReleaseToNannyTask.on_release(self, additional_parameters)
        self.mark_released_resources('stable', ttl=180)

        nanny_client = nanny.NannyClient(
            api_url='http://nanny.yandex-team.ru/', \
            oauth_token=self.get_vault_data(self.owner, 'nanny_robot_oauth_token'))

        nanny_client.update_service_sandbox_file(
            service_id='parallel_biathlon',
            task_type=self.type,
            task_id=str(self.id),
            deploy=True)


__Task__ = GenerateSportDeviceIds
