# -*- coding: utf-8 -*-

import itertools
import random
import re

from sandbox.projects import resource_types
from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.svn import Arcadia

from sandbox.projects.common.thumbdaemon.utils import get_thumb_ids
from sandbox.projects.images.daemons import resources as images_daemons_resources


class DatabaseParameter(parameters.LastReleasedResource):
    name = 'database_resource_id'
    description = 'Database:'
    resource_type = resource_types.VIDEO_ULTRA_THUMB_DATABASE


class RequestLimitParameter(parameters.SandboxIntegerParameter):
    name = 'request_limit'
    description = 'Unique requests number upper limit:'
    default_value = 100000


class ThreadParams(object):
    def __init__(self, **kwargs):
        self.__dict__.update(kwargs)


class GenerateThumbDaemonRequests(SandboxTask):
    """
        Дампает базу тумбнейлов и создает http-запросы к ней
    """
    type = 'GENERATE_THUMB_DAEMON_REQUESTS'
    input_parameters = (DatabaseParameter, RequestLimitParameter)

    def get_result_path(self):
        return self.abs_path('thumb.requests')

    def on_enqueue(self):
        SandboxTask.on_enqueue(self)

        self._create_resource(
            self.descr,
            self.get_result_path(),
            resource_types.THUMB_DAEMON_REQUESTS,
            arch="any")

    def on_execute(self):
        self.sync_resource(self.ctx[DatabaseParameter.name])
        database_resource = channel.sandbox.get_resource(self.ctx[DatabaseParameter.name])
        random.seed(database_resource.file_md5)

        # Get all transformations supported by thumbdaemon
        # Getting from trunk is good enough - non-existent transforms will be handled as default
        transforms = []
        thumb_config_svn_path = Arcadia.trunk_url(images_daemons_resources.NAIL_DAEMON_CONFIG.arcadia_path)

        thumb_config_local_path = images_daemons_resources.NAIL_DAEMON_CONFIG.arcadia_path.split('/')[-1]
        Arcadia.export(thumb_config_svn_path, thumb_config_local_path)
        with open(thumb_config_local_path, 'r') as thumb_config_file:
            for line in thumb_config_file:
                if "ThumbChunk" in line:
                    transform = int(re.match('ThumbChunk:(\d+)=\d+', line).group(1))
                    transforms.append(transform)

        if len(transforms) == 0:
            raise SandboxTaskFailureError("Could not extract transforms from config")

        with open(self.get_result_path(), 'w') as result_file:
            for thumb_id in itertools.islice(get_thumb_ids(self.ctx[DatabaseParameter.name]),
                                             self.ctx[RequestLimitParameter.name]):
                signature_request_ratio = 100
                if random.randrange(signature_request_ratio) % signature_request_ratio == 0:
                    signature_flags = 2**15 - 2  # Turn on all possible signatures, except build-info-related
                    result_file.write("/i?id=%s&sigs=%d\n" % (thumb_id, signature_flags))
                else:
                    n = random.choice(transforms)
                    result_file.write("/i?id=%s&n=%d\n" % (thumb_id, n))


__Task__ = GenerateThumbDaemonRequests
