# -*- coding: utf-8 -*-

import logging
import re

import sandbox.projects.websearch.middlesearch.resources as ms_resources
from sandbox.projects.common.middlesearch.response_patcher import response_patcher
from sandbox.projects.common.fusion.response_patcher import response_patcher as fusion_response_patcher
from sandbox.projects import resource_types
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk import parameters as sp
from sandbox.projects.common.base_search_quality import threadPool
from sandbox.projects.common.base_search_quality import response_saver
from sandbox.projects.common.search.components.cproxy import get_cproxy
from sandbox.projects.common.search import components as sc
from sandbox.projects.common.search import metadebug
from sandbox.sandboxsdk.task import SandboxTask

from sandbox.sandboxsdk.errors import SandboxTaskFailureError
import traceback

_Basesearch1Params = sc.create_fusion_params(1, old_task=True)
_Basesearch2Params = sc.create_fusion_params(2, old_task=True)


BASE1_PORT = sc.DEFAULT_BASESEARCH_PORT
BASE2_PORT = sc.DEFAULT_BASESEARCH_PORT + 999

PARAMS = _Basesearch1Params.params + _Basesearch2Params.params + sc.DefaultMiddlesearchParams.params


_RESPONSE_SAVER_PARAMS = response_saver.create_response_saver_params(
    queries_resource=[
        resource_types.PLAIN_TEXT_QUERIES,
        ms_resources.WebMiddlesearchPlainTextQueries,
        resource_types.IMAGES_MIDDLESEARCH_PLAIN_TEXT_REQUESTS,
    ]
)

results_template = """\
<table border = "1" cellpadding = "6px" cellspacing = "0px">
    <tr>
        <th>
            Total responses
        </th>
        <th>
            GotError: YES
        </th>
        <th>
            GotError: NO
        </th>
        <th>
            TotalDocCount: 0
        </th>
        <th>
            TotalDocCount > 0
        </th>
        </tr>
    <tr>
        <td>
            {total_responses}
        </td>
        <td>
            {errors}
        </td>
        <td>
            {no_errors}
        </td>
        <td>
            {no_docs}
        </td>
        <td>
            {got_docs}
        </td>
</tr>
</table>"""

not_ready_template = "<h4><font color='gray'> GetFusionMiddlesearchResponses task is not finished yet </font></h4>"
error_template = "<h4><font color='red'> Unexpected state of GetFusionMiddlesearchResponses task </font></h4>"


class UseAsyncSearch(sp.SandboxBoolParameter):
    name = 'use_async_search'
    description = 'Use AsyncSearch'
    default_value = True


class FirstBasesearchType(sp.SandboxStringParameter):
    name = 'first_basesearch_type'
    description = '1st basesearch type (should match Route rules, see also SEARCH-1173)'
    default_value = 'FUSION_KUBR'


class SecondBasesearchType(sp.SandboxStringParameter):
    name = 'second_basesearch_type'
    description = '2nd basesearch type (should match Route rules, see also SEARCH-1173)'
    default_value = 'FUSION_KUBR'


class GetFusionMiddlesearchResponses(SandboxTask):
    """
        Analogue of GetMiddlesearchResponses with two
        fusion instances as basesearches:

        Starts 1 middlesearch and 2 basesearch (fusion) instances
        save responses to PLAIN_TEXT_QUERIES in resources:
        BASESEARCH_HR_RESPONSES or SEARCH_PROTO_RESPONSES

        Additionally it verifies the number of TotalDocCount and 'GotError' entries in responses.
    """

    type = 'GET_FUSION_MIDDLESEARCH_RESPONSES'

    execution_space = 150000

    input_parameters = (
        PARAMS +
        _RESPONSE_SAVER_PARAMS.params +
        (
            UseAsyncSearch,
            FirstBasesearchType,
            SecondBasesearchType,
        ) +
        metadebug.PARAMS.params +
        threadPool.PARAMS
    )

    def on_execute(self):
        basesearch1 = sc.get_fusion_search(params=_Basesearch1Params, port=BASE1_PORT)
        basesearch2 = sc.get_fusion_search(params=_Basesearch2Params, port=BASE2_PORT)

        middlesearch = self._create_middlesearch_component()

        try:
            logging.info("-->Starting 1")
            basesearch1.start()

            logging.info("-->Starting 2")
            basesearch2.start()
            basesearch1.wait()
            basesearch2.wait()
            logging.info("-->Starting Middlesearch")

            self._use_middlesearch_component(middlesearch)
        finally:
            try:
                basesearch2.stop()
                basesearch1.stop()
            finally:
                try:
                    self.check_responses()
                except Exception:
                    logging.error("Failed to check responses %s" % traceback.format_exc())

    def on_enqueue(self):
        SandboxTask.on_enqueue(self)

        channel.task = self
        response_saver.create_resources(self)

    def _create_middlesearch_component(self):
        base_port1, base_port2 = self._get_basesearch_ports_for_middlesearch()

        params = self._get_middlesearch_additional_params()

        first_base_type = self.ctx.get(FirstBasesearchType.name, FirstBasesearchType.default_value)
        second_base_type = self.ctx.get(SecondBasesearchType.name, SecondBasesearchType.default_value)
        middlesearch = sc.get_middlesearch(
            basesearches=[
                {
                    'basesearch_type': first_base_type,
                    'hosts_and_ports': [('localhost', base_port1)],
                    'collection': '',
                },
                {
                    'basesearch_type': second_base_type,
                    'hosts_and_ports': [('localhost', base_port2)],
                    'collection': '',
                }
            ],
            disable_timeouts=True,
            **params
        )

        return middlesearch

    def _get_basesearch_ports_for_middlesearch(self):
        if not metadebug.use_cproxy(self.ctx):
            return BASE1_PORT, BASE2_PORT

        proxy_port1 = BASE1_PORT + 2
        proxy_port2 = BASE2_PORT + 2

        proxy = {
            proxy_port1: "localhost:%s" % BASE1_PORT,
            proxy_port2: "localhost:%s" % BASE2_PORT
        }
        cproxy = get_cproxy(self, proxy)
        cproxy.start()
        cproxy.wait_port()

        return proxy_port1, proxy_port2

    def _get_middlesearch_additional_params(self):
        return {
            'use_async_search': self.ctx.get(UseAsyncSearch.name, UseAsyncSearch.default_value),
            'set_n_groups_for_source_multiplier': True,
        }

    def check_responses(self):
        got_error = re.compile("GotError: (YES|NO)")
        docs = re.compile("TotalDocCount: (\d+)")
        ACCEPTABLE_NOT_FOUND_RATE = 25
        ACCEPTABLE_ERRORS_COUNT = 0
        if self.out_resource:
            with open(self.out_resource.path) as responses_file:
                responses = responses_file.read()
                error_stats = got_error.findall(responses)
                self.ctx['errors'] = error_stats.count("YES")
                self.ctx['no_errors'] = error_stats.count("NO")
                doc_stats = docs.findall(responses)
                simple_found = [value for n, value in enumerate(doc_stats) if not n % 3]
                self.ctx["total_responses"] = len(simple_found)
                self.ctx["no_docs"] = simple_found.count("0")
                if self.ctx['total_responses']:
                    self.ctx["not found_rate"] = float(self.ctx["no_docs"]) * 100 / self.ctx['total_responses']
                else:
                    self.ctx['not_found_rate'] = 100
            not_found_rate = self.ctx.get('not found_rate', 1)
            if not_found_rate > ACCEPTABLE_NOT_FOUND_RATE:
                error_text = 'NotFound rate is not acceptable: %s%% > %s%%(%s Errors). See responses for extra info'
                raise SandboxTaskFailureError(error_text %
                                              (not_found_rate, ACCEPTABLE_NOT_FOUND_RATE, self.ctx['errors']))
            if self.ctx['errors'] > ACCEPTABLE_ERRORS_COUNT:
                error_text = 'Errors number is not acceptable: %s > %s (%s NotFound). See responses for extra info'
                raise SandboxTaskFailureError(error_text %
                                              (self.ctx['errors'], ACCEPTABLE_ERRORS_COUNT, self.ctx['no_docs']))

        else:
            raise SandboxTaskFailureError('No responses resource created in %s' % self.out_resource.path)

    def get_results_table(self):
        ctx = self.ctx.copy()
        if self.is_completed():
            if ctx.get('total_responses') is None:
                return error_template
            ctx['got_docs'] = ctx.get('total_responses') - ctx.get('no_docs', 0)
            return results_template.format(**ctx)
        else:
            return not_ready_template

    def _use_middlesearch_component(self, middlesearch):
        def prepare_session():
            if self.ctx[_RESPONSE_SAVER_PARAMS.TestWithCacheDisabled.name]:
                middlesearch.disable_cache()

        logging.info("Fetching queries")
        patched_queries = response_saver.get_patched_queries_resource(self)
        logging.info("got patched_queries %s", patched_queries)

        def int_patcher(q):
            # int requests should have haha=da
            if '&haha=da' in q:
                return q
            return q + '&haha=da'

        queries_patchers = [metadebug.dump_grouping_patcher]
        if self.ctx.get('use_int', False):
            queries_patchers.append(int_patcher)

        self.tcp_dump_proc = metadebug.start_tcpdump(self.ctx, ports=[BASE1_PORT, BASE2_PORT])

        def save_resp_to_res(res, prepare_session_func=None):
            response_saver.save_responses(
                self.ctx,
                search_component=middlesearch,
                responses_resource=res,
                queries_patchers=queries_patchers,
                patched_queries_resource=patched_queries,
                response_patchers=[response_patcher, fusion_response_patcher],
                prepare_session_callback=prepare_session_func,
                need_dbgrlv=False,
            )

        self.out_resource = channel.sandbox.get_resource(self.ctx['out_resource_id'])

        try:
            #  if need to test responses correctness with cache disabled:
            if self.ctx.get(_RESPONSE_SAVER_PARAMS.TestWithCacheDisabled.name):
                logging.info("Getting responses with disabled cache")
                save_resp_to_res(
                    channel.sandbox.get_resource(self.ctx['without_cache_responses_id']),
                    prepare_session_func=middlesearch.disable_cache,
                )
            logging.info("Getting responses with empty enabled cache")
            save_resp_to_res(
                self.out_resource,
                prepare_session_func=middlesearch.clear_cache,
            )
        except:
            raise
        finally:
            # any problem with responses can raise exception
            # but tcpdump should be stopped before we fail the task
            metadebug.stop_tcpdump(self.tcp_dump_proc)


__Task__ = GetFusionMiddlesearchResponses
