# -*- coding: utf-8 -*-

import logging
import urlparse
import traceback

from sandbox.common import errors
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.parameters import ResourceSelector
from sandbox.sandboxsdk.parameters import SandboxBoolParameter
from sandbox.sandboxsdk.parameters import SandboxStringParameter
from sandbox.sandboxsdk.parameters import Container
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sandboxsdk.environments import SvnEnvironment
from sandbox.projects.common.base_search_quality.response_saver import save_responses
from sandbox.projects.common.base_search_quality.response_saver import DefaultResponseSaverParams
from sandbox.projects.common.base_search_quality.response_saver import UNSTABLE_OUTPUT_KEY
from sandbox.projects.common.base_search_quality import threadPool
from sandbox.projects.common.geosearch.search_components import get_addrs_basesearch_params
from sandbox.projects.common.geosearch.search_components import get_addrs_basesearch_business
from sandbox.projects.common.geosearch.search_components import GetMapsSearchGeocoderParams
# from projects.common.geosearch.search_components import patch_yacare_wrapper
from sandbox.projects.common.geosearch.search_components import get_maps_search_geocoder
# from projects.common.geosearch.search_components import patch_geocoder_index_description
# from projects.common.geosearch.search_components import get_geocoder_indexes_from_description
from sandbox.projects.common.geosearch.search_components import GeoMetaSearchMiddleParams
from sandbox.projects.common.geosearch.search_components import GeoMetaSearchUpperParams
from sandbox.projects.common.geosearch.search_components import get_geometasearch_params_from_ctx
from sandbox.projects.common.geosearch.search_components import GeoMetaSearchUpper, GeoMetaSearchMiddle
from sandbox.projects.common.geosearch.search_components import DEFAULT_GEOMETASEARCH_UPPER_PORT, DEFAULT_GEOMETASEARCH_MIDDLE_PORT
from sandbox.projects.common.geosearch.pkg_install import PackageInstallTask
from sandbox.projects.common.geosearch.response_parser import GetGeosearchCustomNodeTypesDict
from sandbox.projects.common.geosearch.response_parser import GeosearchResponsePatcher
from sandbox.projects.common.geosearch.spellchecker_component import GetSpellcheckerParams, get_spellchecker_component
from sandbox.projects.common.utils import create_misc_resource_and_dir
from sandbox.projects.common.wizard.providers.wizard_provider import WizardProvider, WizardConfigType
from sandbox.projects.common.geosearch.reqans_logs import load_reqans_log


class WizardParams(object):

    class Binary(ResourceSelector):
        name = 'wizard_executable_resource_id'
        description = 'Executable'
        resource_type = 'REMOTE_WIZARD'
        group = 'Wizard parameters'
        required = True

    params = (Binary, )


MapsSearchBusinessParams = get_addrs_basesearch_params(ctx_suffix='_business')
MapsSearchGeocoderParams = GetMapsSearchGeocoderParams(ctx_suffix='_geocoder')

SpellcheckerParams = GetSpellcheckerParams()


class ExecutionParams(object):

    def create_use_parameters(
        param_name=None,
        param_description=None,
        component_params=None,
        is_required=False
    ):
        class UrlParameter(SandboxStringParameter):
            name = param_name + '_url'
            description = param_description + ': url'
            group = 'Execution parameters'
            required = True

        class UseParameter(SandboxStringParameter):
            name = param_name
            description = param_description
            group = 'Execution parameters'
            choices = [
                ('Use url', 'use_url'),
                ('Use component', 'use_component'),
            ]
            default_value = 'use_component'
            required = is_required
            sub_fields = {
                'use_url': [UrlParameter.name],
                'use_component': [p.name for p in component_params],
            }

        return UseParameter, UrlParameter

    UseWizard, UseWizardUrl = create_use_parameters(
        param_name='use_wizard',
        param_description='Use wizard',
        component_params=WizardParams.params,
        is_required=True,
    )

    UseMapsSearchBusiness, UseMapsSearchBusinessUrl = create_use_parameters(
        param_name='use_maps_search_business',
        param_description='Use maps-search-business',
        component_params=MapsSearchBusinessParams.params,
    )

    UseMapsSearchGeocoder, UseMapsSearchGeocoderUrl = create_use_parameters(
        param_name='use_maps_search_geocoder',
        param_description='Use maps-search-geocoder',
        component_params=MapsSearchGeocoderParams.params,
    )

    UseSpellchecker, UseSpellcheckerUrl = create_use_parameters(
        param_name='use_spellchecker',
        param_description='Use spellchecker',
        component_params=SpellcheckerParams.params,
    )

    class UseGeometasearchUpper(SandboxBoolParameter):
        name = 'use_geometasearch_upper'
        description = 'Use upper geometasearch'
        group = 'Execution parameters'
        default_value = True
        sub_fields = {
            'true': [p.name for p in GeoMetaSearchUpperParams.params],
        }

    class RestartOnFail(SandboxBoolParameter):
        name = 'restart_on_fail'
        description = 'Try to restart on fail'
        group = 'Execution parameters'
        default_value = False

    class LXCContainer(Container):
        name = 'lxc_container'
        description = 'Container with preinstalled geocoder'
        required = True
        default_value = '306364329'

    params = (
        UseWizard, UseWizardUrl,

        UseMapsSearchBusiness,
        UseMapsSearchBusinessUrl,

        UseMapsSearchGeocoder,
        UseMapsSearchGeocoderUrl,

        UseSpellchecker,
        UseSpellcheckerUrl,

        UseGeometasearchUpper,
        RestartOnFail,

        LXCContainer,
    )


class GetGeometasearchResponses(PackageInstallTask):
    """
        Get GetGeometasearchResponses
    """
    type = 'GET_GEOMETASEARCH_RESPONSES'

    privileged = True
    required_ram = 80 << 10

    # client_tags = ctc.Tag.LINUX_LUCID
    # cores = 17

    input_parameters = (
        DefaultResponseSaverParams.params +
        threadPool.PARAMS +
        # PackageInstallTaskParams.params +
        ExecutionParams.params +
        GeoMetaSearchUpperParams.params +
        GeoMetaSearchMiddleParams.params +
        WizardParams.params +
        MapsSearchBusinessParams.params +
        MapsSearchGeocoderParams.params +
        SpellcheckerParams.params
    )

    environment = (SvnEnvironment(), )

    def on_enqueue(self):
        SandboxTask.on_enqueue(self)

        self.ctx['out_resource_id'] = self._create_resource(
            self.descr,
            'responses.txt',
            'BASESEARCH_HR_RESPONSES',
            arch='any'
        ).id

        self.ctx['patched_queries_resource_id'] = self._create_resource(
            'patched queries',
            'patched_queries.txt',
            'PLAIN_TEXT_QUERIES',
            arch='any'
        ).id

    def param_value(self, param):
        return self.ctx.get(param.name)

    def normalize_url(self, url, collection=''):
        o = urlparse.urlparse(url)
        return urlparse.urlunparse((o.scheme, o.netloc, collection, '', '', ''))

    def on_prepare(self):
        # Prepare arcadia dependent parts
        self.middle_params = get_geometasearch_params_from_ctx(
            task=self, search_type='middle', params=GeoMetaSearchMiddleParams)
        self.upper_params = get_geometasearch_params_from_ctx(
            task=self, search_type='upper', params=GeoMetaSearchUpperParams)
        # #patch_yacare_wrapper()
        # index_description_tmp = channel.task.sync_resource(self.ctx.get('maps_search_geocoder_index_description_resource_id_geocoder'))
        # index_description_path = './index_description.json'
        # copy_path(index_description_tmp, index_description_path)
        # self.patched_geocoder_index_description = patch_geocoder_index_description(index_description_path)
        # #get_geocoder_indexes_from_description(self.patched_geocoder_index_description)
        # get_geocoder_indexes_from_description(self.patched_geocoder_index_description)

    def on_execute(self):
        # self.install_packages()

        restart_on_fail = self.ctx.get(ExecutionParams.RestartOnFail.name)
        if restart_on_fail:
            num_tries_left = 2
        else:
            num_tries_left = 1

        while num_tries_left > 0:
            try:
                self.get_responses()
                num_tries_left = 0
            except errors.TaskFailure:
                num_tries_left -= 1
                self.ctx['was_fail'] = 1
                if num_tries_left == 0 or self.ctx.get(UNSTABLE_OUTPUT_KEY, False):
                    raise
                logging.info('Task is failed. Another try.')

    def get_short_task_result(self):
        if self.is_completed():
            if self.ctx.get('was_fail'):
                return "was fail"
            else:
                return "ok"
        return None

    def get_responses(self):
        # Basic searches
        if not self.param_value(ExecutionParams.UseMapsSearchGeocoder) == 'use_component':
            maps_search_geocoder = None
        else:
            maps_search_geocoder = get_maps_search_geocoder(
                ctx=self.ctx,
                # index_description=self.index_description_path,
                index_description=self.patched_geocoder_index_description,
                params=MapsSearchGeocoderParams,
            )

        if not self.param_value(ExecutionParams.UseMapsSearchBusiness) == 'use_component':
            maps_search_business = None
        else:
            maps_search_business = get_addrs_basesearch_business(
                ctx=self.ctx,
                params=MapsSearchBusinessParams,
            )

        # Prepare Spellchecker
        if not self.param_value(ExecutionParams.UseSpellchecker) == 'use_component':
            spellchecker = None
        else:
            configs_dir = create_misc_resource_and_dir(
                self.ctx, 'patched_configs_resource_id', 'patched configs', 'configs'
            )
            spellchecker = get_spellchecker_component(
                ctx=self.ctx,
                params=GetSpellcheckerParams(),
                configs_dir=configs_dir,
            )

        # Prepare wizard
        if not self.param_value(ExecutionParams.UseWizard) == 'use_component':
            wizard = None
        else:
            self.sync_resource(self.ctx[WizardParams.Binary.name])
            wizard_res = channel.sandbox.get_resource(
                self.ctx[WizardParams.Binary.name])
            build_wizard_task = channel.sandbox.get_task(wizard_res.task_id)
            if build_wizard_task.type != 'BUILD_WIZARD':
                raise errors.TaskFailure('Wizard must be created with BUILD_WIZARD task')

            wizard = WizardProvider(wizard_res.task_id,
                                    config_type=WizardConfigType.geo(),
                                    work_with_runtime=False,
                                    read_only_resources=True
                                    )

        # Prepare middle
        hostname = 'localhost'
        searchsources_middle = {}
        if maps_search_business:
            searchsources_middle[
                'Business'] = 'http://%s:%s/' % (hostname, str(maps_search_business.port))
        elif self.param_value(ExecutionParams.UseMapsSearchBusiness) == 'use_url':
            url = self.param_value(ExecutionParams.UseMapsSearchBusinessUrl)
            searchsources_middle['Business'] = self.normalize_url(url)

        if 'Business' in searchsources_middle:
            searchsources_middle[
                'BusinessGeo'] = searchsources_middle['Business']
            searchsources_middle[
                'BusinessAdvert'] = searchsources_middle['Business']
            searchsources_middle[
                'BusinessGeoAdvert'] = searchsources_middle['Business']
            searchsources_middle[
                'BusinessNav'] = searchsources_middle['Business']
            searchsources_middle[
                'PermalinksInfo'] = searchsources_middle['Business']

        if maps_search_geocoder:
            searchsources_middle[
                'Geocoder'] = 'http://%s:%s/maps' % (hostname, str(maps_search_geocoder.port))
        elif self.param_value(ExecutionParams.UseMapsSearchGeocoder) == 'use_url':
            url = self.param_value(ExecutionParams.UseMapsSearchGeocoderUrl)
            searchsources_middle['Geocoder'] = self.normalize_url(
                url, collection='maps')

        if wizard:
            wizard_url = 'localhost:{0}'.format(wizard.port)
        elif self.param_value(ExecutionParams.UseWizard) == 'use_url':
            wizard_url = self.param_value(ExecutionParams.UseWizardUrl)

        out_resource = channel.sandbox.get_resource(
            self.ctx['out_resource_id'])
        patched_queries = channel.sandbox.get_resource(
            self.ctx['patched_queries_resource_id'])

        middle_output = self.create_resource(
            'middle stdout/stderr',
            'geometasearch_middle_output.txt',
            'GEOMETASEARCH_OUTPUT',
            arch='any'
        )
        self.ctx['middle_output_resource_id'] = middle_output.id

        middle = GeoMetaSearchMiddle(
            work_dir=channel.task.abs_path(),
            searchsources=searchsources_middle,
            port=DEFAULT_GEOMETASEARCH_MIDDLE_PORT,
            event_log=None,
            load_log=None,
            wizard=wizard_url,
            output_resource=middle_output,
            **self.middle_params)

        # Prepare upper
        use_upper = self.ctx.get(ExecutionParams.UseGeometasearchUpper.name)
        if not use_upper:
            upper = None
        else:
            auxsources_upper = {}
            searchsources_upper = {}
            searchsources_upper[
                'Geo'] = 'http://%s:%s/yandsearch' % (hostname, str(middle.port))
            searchsources_upper['GeoMisspell'] = searchsources_upper['Geo']
            if spellchecker:
                auxsources_upper[
                    'Misspell'] = 'http://%s:%s/yandsearch' % (hostname, str(spellchecker.port))
            elif self.param_value(ExecutionParams.UseSpellchecker) == 'use_url':
                url = self.param_value(ExecutionParams.UseSpellcheckerUrl)
                auxsources_upper['Misspell'] = self.normalize_url(url)

            reqans_log = self.create_resource(
                'reqans log',
                'geometasearch_reqans.log',
                'GEOMETASEARCH_REQANS_LOG',
                arch='any'
            )
            self.ctx['reqans_log_resource_id'] = reqans_log.id

            upper_output = self.create_resource(
                'upper stdout/stderr',
                'geometasearch_upper_output.txt',
                'GEOMETASEARCH_OUTPUT',
                arch='any'
            )
            self.ctx['upper_output_resource_id'] = upper_output.id

        upper = GeoMetaSearchUpper(
            work_dir=channel.task.abs_path(),
            searchsources=searchsources_upper,
            auxsources=auxsources_upper,
            port=DEFAULT_GEOMETASEARCH_UPPER_PORT,
            event_log=None,
            load_log=None,
            reqans_log_resource=reqans_log,
            output_resource=upper_output,
            **self.upper_params
        )

        # Starting
        tracebacks = []

        components = [
            maps_search_geocoder,
            maps_search_business,
            spellchecker,
        ]

        if use_upper:
            main_component = upper
            components.append(middle)
        else:
            main_component = middle

        started_components = []

        def start_components():
            logging.info('Starting geosearch components')
            if wizard:
                wizard.start()
            for component in components:
                if component:
                    component.start()
                    started_components.append(component)

            for component in components:
                if component:
                    component.wait()

        def stop_components():
            logging.info('Stopping geosearch components')
            if wizard:
                try:
                    # check for wizard alive. other components checks this when
                    # stop() called
                    if not wizard.alive():
                        raise errors.TaskFailure('wizard is dead')
                    wizard.stop()
                except errors.TaskFailure:
                    tracebacks.append(traceback.format_exc())

            for component in started_components:
                if component:
                    try:
                        component.stop()
                    except errors.TaskFailure:
                        tracebacks.append(traceback.format_exc())

            del started_components[:]

        try:
            query_counter = [1]

            def patch_query(q):
                if not q.startswith('?'):
                    qparts = q.split('&')
                else:
                    qparts = q[1:].split('&')

                # Remove unwelcome params.
                unwelcome_params = (
                    'ask_direct',
                    'ask_counters',
                    'shufadv',
                    'parent_reqid'
                )

                qparts = filter(
                    lambda p: all([not p.startswith(n + '=') for n in unwelcome_params]), qparts)
                q = '?' + '&'.join(qparts)

                # Add testing params.
                testing_params = [
                    'hr=da',
                    'nocache=da',
                    'shufadv=0',
                    'advertseed=1',
                    'dump=request',
                    'search_experimental_nocache=da',
                    'search_experimental_waitall=da',
                    'relev_dump_query_type_factors=1',
                    'waitall=da',
                    'parent_reqid={}'.format(query_counter[0]),
                    'reqid={}'.format(query_counter[0]),
                ]
                q += '&' + '&'.join(testing_params)

                query_counter[0] += 1

                return q

            def add_timestamp_to_relev(q):
                return q + "&relev_ts=1439467641"

            try:
                save_responses(
                    self.ctx,
                    search_component=main_component,
                    responses_resource=out_resource,
                    queries_patchers=[patch_query, add_timestamp_to_relev],
                    ignore_empty_response=True,
                    custom_node_types_dict=GetGeosearchCustomNodeTypesDict(),
                    response_patchers=[GeosearchResponsePatcher],
                    transform_if_response_is_xml=True,
                    save_unpatched_responses=True,
                    patched_queries_resource=patched_queries,
                    on_start_get_responses=start_components,
                    on_end_get_responses=stop_components,
                )
            finally:
                # in case response_saver didn't
                if main_component.is_running():
                    main_component.stop()

            # We should perform all checks, that we need for proper compare
            # Becuase in the case of broken logs we should fail at GetResponses, not at Compare
            # cause TestEnv can't iterpert failed Compare as bad or broken revision,
            # just wrong internal behaviour of comparer
            def _check_reqans_logs(resource_logs):
                try:
                    load_reqans_log(resource_logs.path)
                except Exception, e:
                    raise errors.TaskFailure('BROKEN ReqAns: %s' % e)

            logging.info('Start reqans checking')

            if reqans_log is not None:
                _check_reqans_logs(reqans_log)

            logging.info('Stop reqans checking')
        except:  # Fail on any
            tracebacks.append(traceback.format_exc())

        if len(tracebacks) == 1:
            raise errors.TaskFailure(
                'Error occurred during get responses. \n' + tracebacks[0])
        elif len(tracebacks) > 1:
            msg = 'Exceptions catched during get responses: \n'
            for i, tb in enumerate(tracebacks):
                msg += 'Exception #%d\n%s\n' % (i, tb)
            logging.info(msg)
            raise errors.TaskFailure(msg)


__Task__ = GetGeometasearchResponses
