# -*- coding: utf-8 -*-
import json
import logging
import os
import socket
import time
from contextlib import nested
from urlparse import parse_qsl

import requests

from sandbox import sdk2

from sandbox.projects import resource_types
from sandbox.projects.common import decorators
from sandbox.projects.common import file_utils as fu
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common.search import components as sc
from sandbox.projects.common.search.response import cgi
from sandbox.projects.common.wizard import resources as wr
from sandbox.projects.websearch.begemot import AllBegemotServices as ABS
from sandbox.projects.websearch.begemot import resources as br


_SEARCH_APP_HOST_SOURCES = 'search.app_host.sources.*'
_JSON_DUMP = '{saps}.(_.name eq "{component}")'


class GetHamsterApphostResponses(sdk2.Task):

    class Parameters(sdk2.Task.Parameters):
        max_restarts = 10
        kill_timeout = 5400
        execution_space = 40 * 1024  # 40 Gb

        upper_host = sdk2.parameters.String('Uppersearch host', default='https://hamster.yandex.ru')
        text_requests = sdk2.parameters.Resource('Text requests', required=True, resource_type=wr.WizardQueries)
        limit_requests = sdk2.parameters.Integer('Limit requests', default=0)
        sources_timeout = sdk2.parameters.Integer('Add timeout for sources in ms()', default=0)

        custom_cgi_params = sdk2.parameters.String('Custom cgi parameters', required=False, default='')
        component = sdk2.parameters.String('Search apphost sources component', required=True)

        wizard = sdk2.parameters.Bool('Use custom wizard')
        begemot_merger = sdk2.parameters.Bool('Use custom begemot merger')
        begemot_worker = sdk2.parameters.Bool('Use custom begemot worker')

        with wizard.value[True]:
            wizard_block = sdk2.parameters.Info('Wizard Resources')
            wizard_executable = sdk2.parameters.Resource(
                'Wizard executable resource id', required=True, resource_type=resource_types.REMOTE_WIZARD
            )
            wizard_data = sdk2.parameters.Resource(
                'Wizard data resource id', required=True, resource_type=resource_types.WIZARD_SHARD
            )
            wizard_config = sdk2.parameters.Resource(
                'Wizard config resource id',
                required=True,
                resource_type=resource_types.WIZARD_CONFIG
            )
            wizard_package = sdk2.parameters.Resource(
                'Wizard runtime package resource',
                required=True,
                resource_type=resource_types.WIZARD_RUNTIME_PACKAGE
            )

        with begemot_merger.value[True]:
            bmerger_block = sdk2.parameters.Info('Begemot merger Resources')
            bm_executable = sdk2.parameters.Resource(
                'Begemot Merger executable resource id', required=True, resource_type=br.BEGEMOT_EXECUTABLE
            )
            bm_config = sdk2.parameters.Resource(
                'Begemot Merger config resource id',
                required=True,
                resource_type=[br.BEGEMOT_CONFIG_WITH_CGI, br.BEGEMOT_CONFIG]
            )
            bm_data = sdk2.parameters.Resource(
                'Begemot Merger shard data', required=True, resource_type=br.BEGEMOT_ISS_MERGER
            )

        with begemot_worker.value[True]:
            bworker_block = sdk2.parameters.Info('Begemot worker Resources')
            bw_executable = sdk2.parameters.Resource(
                'Begemot worker executable resource id', required=True, resource_type=br.BEGEMOT_EXECUTABLE
            )
            bw_config = sdk2.parameters.Resource(
                'Begemot worker config resource id',
                required=True,
                resource_type=[br.BEGEMOT_CONFIG_WITH_CGI, br.BEGEMOT_CONFIG]
            )
            bw_data = sdk2.parameters.Resource(
                'Begemot Wizard shard data',
                required=True,
                resource_type=[s.data_resource_type for s in ABS.Service.itervalues() if s.data_resource_type is not None]
            )

    class Context(sdk2.Task.Context):
        rps = 0
        response_id = None

    def _prepare_requests_info(self):
        queries_res = sdk2.ResourceData(self.Parameters.text_requests)
        return self._parse_user_queries(queries_res)

    @staticmethod
    def _parse_user_queries(queries_res):
        for line in fu.read_line_by_line(str(queries_res.path)):
            if line.startswith('@'):
                continue
            req, _, rules = line.partition('$print:')
            req, s, param = req.partition('$cgi:')
            if not s:
                rules, _, param = rules.partition('$cgi:')
            request_params = parse_qsl(param)
            request_params.append(('text', req))
            yield request_params

    def on_enqueue(self):
        self.Context.response_id = resource_types.PLAIN_TEXT_QUERIES(self, 'AppHost responses', 'responses.txt').id

    def on_execute(self):
        req_info = self._prepare_requests_info()
        workers = []
        srcrwr = []

        if self.Parameters.wizard:
            wizard_data_path = str(sdk2.ResourceData(self.Parameters.wizard_data).path)
            wizard_binary_path = str(sdk2.ResourceData(self.Parameters.wizard_executable).path)
            wizard_config_path = str(sdk2.ResourceData(self.Parameters.wizard_config).path)
            runtime_data_path = str(sdk2.ResourceData(self.Parameters.wizard_package).path)
            wizard = sc.get_wizard(
                wizard_binary_path, wizard_config_path, wizard_data_path, runtime_data_path, task=self
            )
            workers.append(wizard)
            srcrwr.append('WIZARD:{host}:{port}'.format(host=socket.gethostname(), port=wizard.apphost_port))

        if self.Parameters.begemot_merger:
            bmerger_binary = str(sdk2.ResourceData(self.Parameters.bm_executable).path)
            bmerger_config = self._get_begemot_config_path(self.Parameters.bm_config)
            bmerger_data = str(sdk2.ResourceData(self.Parameters.bm_data).path)
            bmerger = sc.get_begemot(bmerger_binary, bmerger_config, port=31400, task=self, worker_dir=bmerger_data)
            workers.append(bmerger)
            srcrwr.append('BEGEMOT_MERGER:{host}:{port}'.format(host=socket.gethostname(), port=bmerger.port))

        if self.Parameters.begemot_worker:
            bworker_binary = str(sdk2.ResourceData(self.Parameters.bw_executable).path)
            bworker_config = self._get_begemot_config_path(self.Parameters.bw_config)
            bworker_data = str(sdk2.ResourceData(self.Parameters.bw_data).path)
            bworker = sc.get_begemot(bworker_binary, bworker_config, port=31401, task=self, worker_dir=bworker_data)
            workers.append(bworker)
            srcrwr.append('BEGEMOT_WORKER_P:{host}:{port}'.format(host=socket.gethostname(), port=bworker.port))

        self._get_responses(req_info, workers, srcrwr)

    @decorators.retries(3)
    def _get_json_response(self, url):
        r = requests.get(url.base_url, params=url.params, verify=False)
        return r.json()

    @staticmethod
    def _get_begemot_config_path(config_parameter):
        config_path = str(sdk2.ResourceData(config_parameter).path)
        if config_parameter.type == str(br.BEGEMOT_CONFIG):
            return config_path
        return os.path.join(config_path, 'worker.cfg')

    def _get_responses(self, req_info, workers, srcrwr):
        with nested(* workers):
            reqs = self._create_requests(req_info, srcrwr_list=srcrwr)
            responses = self._get_fetched_responses(reqs)
        self._save_responses(responses)

    def _create_requests(self, req_info, srcrwr_list):
        """
            :param req_info: generator, so requires only one cycle
        """
        logging.info('Creating requests...')
        common_url = cgi.UrlCgiCustomizer(base_url='{}/search'.format(self.Parameters.upper_host))
        common_url.add_custom_param(
            'json_dump',
            _JSON_DUMP.format(saps=_SEARCH_APP_HOST_SOURCES, component=self.Parameters.component)
        )
        for param in parse_qsl(self.Parameters.custom_cgi_params):
            common_url.add_custom_param(param[0], param[1])
        for srcrwr in srcrwr_list:
            if self.Parameters.sources_timeout > 0:
                srcrwr += ':{}'.format(self.Parameters.sources_timeout)
            common_url.add_custom_param('srcrwr', srcrwr)
        reqs = []
        for req in req_info:
            iter_url = cgi.UrlCgiCustomizer(common_url.base_url, common_url.params)
            for param in req:
                iter_url.add_custom_param(param[0], param[1])
            reqs.append(iter_url)
        return reqs

    def _get_fetched_responses(self, reqs):
        logging.info('Dumping responses...')
        start_time = time.time()
        responses = []
        fail_count = 0
        for url in reqs:
            try:
                resp = self._get_json_response(url)
                if _SEARCH_APP_HOST_SOURCES not in resp:
                    logging.error('No requested searcher props in response:\n{}\n----------'.format(resp))
                    continue
                for props in resp[_SEARCH_APP_HOST_SOURCES]:
                    if 'results' not in props:
                        logging.warning('Empty results detected!\nUrl: {}'.format(url))
                        fail_count += 1
                        break
                    responses.append(json.dumps(props['results'][0]))
                if self.Parameters.limit_requests and len(responses) >= self.Parameters.limit_requests:
                    break
            except Exception:
                logging.error(
                    '\n\n==================\nError during request processing:\n{0}\nUrl: {1}\n'
                    '\n==================\n\n'.format(eh.shifted_traceback(), url)
                )
        time_delta = time.time() - start_time

        if time_delta < 0.1:
            time_delta = 0.1
        rps = len(reqs) / time_delta
        self.Context.rps = rps
        eh.ensure(responses, 'No results were collected')
        logging.info('Fetched requests with {0} rps, with {1} fail(s)'.format(rps, fail_count))

        return responses

    def _save_responses(self, responses):
        logging.info('Saving responses ...')
        fu.write_lines(str(sdk2.Resource[self.Context.response_id].path), responses)
