import os
import logging

from os.path import join
from sandbox.projects import resource_types
from sandbox.sandboxsdk import process
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.paths import copy_path, make_folder
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.parameters import SandboxArcadiaUrlParameter


class GetLemurVintageMapTestData(SandboxTask):
    type = 'GET_LEMUR_VINTAGE_MAP_TEST_DATA'
    token = "AQAD-qJSJqkvAAAA-zwPdYYyLUAogU3pTLaxfes"

    class ArcadiaRootUrl(SandboxArcadiaUrlParameter):
        name = 'arcadia_root_url'
        default_value = Arcadia.trunk_url()
        description = 'Arcadia svn url:'

    input_parameters = [ArcadiaRootUrl]

    def get_revision(self):
        url_split = self.ctx['arcadia_root_url'].split('@')
        revision = "HEAD"

        if len(url_split) > 1:
            revision = url_split[1]

        return revision

    def download_resource(self, resource_id, download_path):
        resource_local_path = channel.task.sync_resource(resource_id)
        copy_path(resource_local_path, download_path)

    def on_execute(self):
        ytio_resource_id = 243757870
        conf_dir = "config"

        logging.debug("Arcadia URL: {}".format(self.ctx['arcadia_root_url']))
        arcadia_url = join(self.ctx['arcadia_root_url'].split('@')[0], "robot/lemur/conf/conf-production")
        conf_path = Arcadia.checkout(arcadia_url, conf_dir, revision=self.get_revision())

        yt_root = "//home/lemur-dev/test_data/prepare_test_data/"
        tables = [
            "data/00000000/delta/incoming",
            "data/00000000/persistent/hostdata",
            "data/00000001/delta/incoming",
            "data/00000001/persistent/hostdata",
            "data/global", "data/00000000/global",
            "data/00000000/persistent/urldata",
            "data/00000001/global",
            "data/00000001/persistent/urldata",
        ]

        new_test_data_dir = "vintage_map_input"
        bin_dir = "bins"
        error_dir = "log_err"
        make_folder(new_test_data_dir)
        make_folder(bin_dir)
        make_folder(error_dir)

        self.download_resource(ytio_resource_id, bin_dir)
        ytio = join(bin_dir, "ytio")

        env = os.environ.copy()
        env['YT_TOKEN'] = self.token

        for t in tables:
            cmd = [ytio, '-i', join(conf_path, "instance_config_yt.pb.txt"), '-t', join(yt_root, t), '-n', '1500']
            logging.debug("CMD: %s", ' '.join(cmd))
            with open(join(new_test_data_dir, t.replace("/", "__")), 'w') as stdout:
                with open(join(error_dir, t.replace("/", "__"))+".err", 'w') as stderr:
                    process.run_process(cmd, check=True, stdout=stdout, stderr=stderr, log_prefix="download_data", outputs_to_one_file=False, wait=True, environment=env)
            logging.info("{} DONE".format(t))

        out_resource = self.create_resource(
            description="{} Task:#{} Revision:{}".format(self.type, self.id, self.get_revision()),
            resource_path=new_test_data_dir,
            resource_type=resource_types.LEMUR_VINTAGE_MAP_TEST_DATA,
        )

        self.mark_resource_ready(out_resource.id)

        subtask = self.create_subtask(
            task_type='TEST_LEMUR_VINTAGE',
            description='TEST_LEMUR_VINTAGE run to check new test data',
            input_parameters={
                'arcadia_root_url': "{}@{}".format(Arcadia.trunk_url(), Arcadia.info(conf_dir)['commit_revision']),
                'input_data_res': out_resource.id,
                'update_input_data': True
            }
        )

        logging.debug("Waiting for subtask: {}".format(subtask.id))
        self.wait_tasks(
            [subtask.id],
            list(self.Status.Group.FINISH + self.Status.Group.BREAK),
            wait_all=True
        )


__Task__ = GetLemurVintageMapTestData
