# -*- coding: utf-8 -*-

from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.channel import channel

from sandbox.projects import resource_types
from sandbox.projects.common.search import components as sc
from sandbox.projects.common.base_search_quality import threadPool
from sandbox.projects.common.base_search_quality import response_saver

_MMETA_PORT = sc.DEFAULT_MIDDLESEARCH_PORT
_INT_PORT = sc.DEFAULT_MIDDLESEARCH_PORT + 1

MMETA_PARAMS = sc.create_middlesearch_params('mmeta', group_name='Mmeta parameters')
INT_PARAMS = sc.create_middlesearch_params('int', group_name='Int parameters')
_RESPONSE_SAVER_PARAMS = response_saver.create_response_saver_params(
    queries_resource=[
        resource_types.PLAIN_TEXT_QUERIES,
        resource_types.IMAGES_MIDDLESEARCH_PLAIN_TEXT_REQUESTS,
        resource_types.VIDEO_MIDDLESEARCH_PLAIN_TEXT_REQUESTS
    ]
)


class GetMmetaAndIntResponses(SandboxTask):
    """
        Starts 1 local middlesearch and 1 local int instances.
        Middlesearch' SearchSource points to local int instance.
        Int uses config without modifying SearchSource sections
        i.e. sends subsource requests to **production** basesearch instances.
        Task saves responses to ``PLAIN_TEXT_QUERIES`` in resource ``BASESEARCH_HR_RESPONSES``.
    """

    type = 'GET_MMETA_AND_INT_RESPONSES'

    input_parameters = (
        MMETA_PARAMS.params +
        INT_PARAMS.params +
        _RESPONSE_SAVER_PARAMS.params +
        threadPool.PARAMS
    )

    def on_enqueue(self):
        SandboxTask.on_enqueue(self)
        channel.task = self

        self.ctx['out_resource_id'] = self.create_resource(
            self.descr, 'responses.txt', resource_types.BASESEARCH_HR_RESPONSES, arch='any').id

        self.ctx['mmeta_eventlog_out_resource_id'] = self.create_resource(
            'mmeta eventlog', 'mmeta_event.log',
            resource_types.EVENTLOG_DUMP,
            arch='any',
            attributes={'mmeta': '1'}
        ).id
        self.ctx['int_eventlog_out_resource_id'] = self.create_resource(
            'int eventlog', 'int_event.log',
            resource_types.EVENTLOG_DUMP,
            arch='any',
            attributes={'int': '1'}
        ).id

    def on_execute(self):
        mmeta_eventlog_resource = channel.sandbox.get_resource(self.ctx['mmeta_eventlog_out_resource_id'])
        int_eventlog_resource = channel.sandbox.get_resource(self.ctx['int_eventlog_out_resource_id'])

        middlesearch_mmeta = sc.get_middlesearch(
            basesearches=[{'hosts_and_ports': [('localhost', _INT_PORT)]}],
            params=MMETA_PARAMS,
            disable_timeouts=False,
            port=_MMETA_PORT,
            event_log=mmeta_eventlog_resource.path,
            work_dir=self.abs_path('mmeta.dir'),
        )

        middlesearch_int = sc.get_middlesearch(
            basesearches=None,
            params=INT_PARAMS,
            disable_timeouts=False,
            port=_INT_PORT,
            event_log=int_eventlog_resource.path,
            work_dir=self.abs_path('int.dir'),
        )

        middlesearch_int.start()
        middlesearch_int.wait()

        self._use_middlesearch_component(middlesearch_mmeta)

        middlesearch_int.stop()

    def _use_middlesearch_component(self, middlesearch_mmeta):
        out_resource = channel.sandbox.get_resource(self.ctx['out_resource_id'])

        response_saver.save_responses(
            self.ctx,
            search_component=middlesearch_mmeta,
            responses_resource=out_resource,
            need_max_timeout=False,
            need_complete_answer=True,
            need_dbgrlv=False,
        )


__Task__ = GetMmetaAndIntResponses
