# -*- coding: utf-8 -*-
import sandbox.projects.websearch.middlesearch.resources as ms_resources
from sandbox.projects import resource_types
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk import parameters as sp
from sandbox.projects.common.search.eventlog import eventlog
from sandbox.projects.common.search import components as sc
from sandbox.projects.common.base_search_quality import threadPool
from sandbox.projects.common.base_search_quality import response_saver
from sandbox.projects.common import utils


_RESPONSE_SAVER_PARAMS = response_saver.create_response_saver_params(
    queries_resource=[
        resource_types.PLAIN_TEXT_QUERIES,
        ms_resources.WebMiddlesearchPlainTextQueries,
        resource_types.IMAGES_MIDDLESEARCH_PLAIN_TEXT_REQUESTS,
        resource_types.VIDEO_MIDDLESEARCH_PLAIN_TEXT_REQUESTS,
    ]
)


class DisableClientTimeoutParameter(sp.SandboxBoolParameter):
    name = 'disable_client_timeout'
    description = 'Disable timeout for socket ops on metasearch'
    default_value = False


class NeedCompleteAnswerParameter(sp.SandboxBoolParameter):
    name = 'need_complete_answer'
    description = 'Retry incomplete answers'
    default_value = True


class GetMmetaResponses(SandboxTask):
    """
        Starts 1 mmeta instance.
        mmeta uses config without modifying SearchSource sections
        i.e. sends subsource requests to production int instances.
        Task saves responses to PLAIN_TEXT_QUERIES in resource BASESEARCH_HR_RESPONSES
    """

    type = 'GET_MMETA_RESPONSES'

    execution_space = 50000  # 50 Gb

    input_parameters = (
        sc.DefaultMiddlesearchParams.params +
        _RESPONSE_SAVER_PARAMS.params +
        threadPool.PARAMS +
        (
            DisableClientTimeoutParameter,
            NeedCompleteAnswerParameter,
        )
    )

    def on_failure(self):
        eventlog.locate_instability_problems(self)
        SandboxTask.on_failure(self)

    def on_enqueue(self):
        SandboxTask.on_enqueue(self)
        response_saver.create_resources(self)

        self.ctx['eventlog_out_resource_id'] = self.create_resource(
            'eventlog', 'event.log', resource_types.EVENTLOG_DUMP, arch='any'
        ).id

    def on_execute(self):
        eventlog_resource = channel.sandbox.get_resource(self.ctx['eventlog_out_resource_id'])

        middlesearch = sc.get_middlesearch(
            basesearches=None,
            disable_timeouts=False,
            event_log=eventlog_resource.path,
            disable_client_timeout=self.ctx[DisableClientTimeoutParameter.name]
        )
        middlesearch.set_cache_thread_limit()  # For SEARCH-1474

        self._use_middlesearch_component(middlesearch)

    def _use_middlesearch_component(self, middlesearch):
        out_resource = channel.sandbox.get_resource(self.ctx['out_resource_id'])

        response_saver.save_responses(
            self.ctx,
            search_component=middlesearch,
            responses_resource=out_resource,
            need_max_timeout=False,
            need_complete_answer=utils.get_or_default(self.ctx, NeedCompleteAnswerParameter),
            need_dbgrlv=False,
        )


__Task__ = GetMmetaResponses
