# -*- coding: utf-8 -*-

import sandbox.common.types.client as ctc
from sandbox.projects.common import utils
from sandbox.projects import resource_types
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk import paths
from sandbox.sandboxsdk import parameters as sp
from sandbox.sandboxsdk import process

from sandbox.projects.common.news.newsd import SlaveNewsd, create_newsd_params


class AppHostQueries(sp.ResourceSelector):
    name = "apphost_queries"
    description = "slave_newsd apphost requests"
    resource_type = ['SLAVE_NEWSD_APPHOST_REQUESTS']
    group = "Newsd apphost params"
    requred = True


class AppHostRequester(sp.ResourceSelector):
    name = "apphost_requester"
    description = "Binary to send apphost requests to slave_newsd"
    resource_type = ['SLAVE_NEWSD_APPHOST_REQUESTER']
    group = "Newsd apphost params"
    requred = False


class StartupWaitTime(sp.SandboxIntegerParameter):
    name = 'startup_wait_time'
    description = "Seconds to wait for newsd to become ready"
    required = False
    default_value = 180


class GetNewsdApphostResponses(SandboxTask):
    """
        Получает и сохраняет результаты обстрела slave_newsd apphost'овыми запросами
    """
    type = 'GET_NEWSD_APPHOST_RESPONSES'

    newsd_params = create_newsd_params()
    input_parameters = \
        newsd_params.params \
        + (AppHostQueries, AppHostRequester, StartupWaitTime)
    client_tags = ctc.Tag.Group.LINUX

    def on_enqueue(self):
        SandboxTask.on_enqueue(self)
        resource = self.create_resource(
            self.descr,
            "newsd_apphost_responses.txt",
            resource_types.SLAVE_NEWSD_APPHOST_RESPONSES
        )
        self.ctx['out_resource_id'] = resource.id

    def on_execute(self):
        out_resource = channel.sandbox.get_resource(self.ctx['out_resource_id'])

        workdir = self.path('work')
        paths.make_folder(workdir, True)
        port = 17171

        timestamp = channel.sandbox.get_resource_attribute(
            self.ctx[self.newsd_params.StateDump.name],
            attribute_name="timestamp",
        )

        tzdata = None
        if self.ctx.get(self.newsd_params.IndexConfig.name):
            tzdata = self.sync_resource(self.ctx[self.newsd_params.IndexConfig.name])

        newsd = SlaveNewsd(
            workdir=workdir,
            binary=self.sync_resource(self.ctx[self.newsd_params.Binary.name]),
            cfg=self.sync_resource(self.ctx[self.newsd_params.Config.name]),
            port=port,
            state=self.sync_resource(self.ctx[self.newsd_params.StateDump.name]),
            geobase=self.sync_resource(self.ctx[self.newsd_params.Geobase.name]),
            index_config_path=self.sync_resource(self.ctx[self.newsd_params.IndexConfig.name]),
            now=timestamp,
            tzdata=tzdata,
        )

        newsd.start()

        requester = None
        if self.ctx.get(AppHostRequester.name):
            requester = self.sync_resource(self.ctx[AppHostRequester.name])
        else:
            resource_id = utils.get_and_check_last_released_resource_id(
                resource_types.SLAVE_NEWSD_APPHOST_REQUESTER,
                arch='linux',
            )
            requester = self.sync_resource(resource_id)
        requests = self.sync_resource(self.ctx[AppHostQueries.name])

        newsd.wait(timeout=self.ctx.get(StartupWaitTime.name, 180))

        cmd = [
            requester,
            '--address', "post://localhost:{}".format(str(port+2)),
            '--input', str(requests),
            '--output', str(out_resource.path),
            ]
        self.process_parameters = cmd

        self.process = process.run_process(cmd, log_prefix='requester_', wait=True)

        self.mark_resource_ready(self.ctx['out_resource_id'])


__Task__ = GetNewsdApphostResponses
