# -*- coding: utf-8 -*-

import os
import datetime

import sandbox.sandboxsdk.environments as env
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk import parameters as sp
from sandbox.sandboxsdk.environments import VirtualEnvironment
from sandbox.sandboxsdk import process

from sandbox.projects.common.search import bugbanner

from sandbox.projects import resource_types


class UidsCountParameter(sp.SandboxIntegerParameter):
    name = 'uids_count'
    description = 'Count of random uids will be generated'
    default_value = 1000


class GetRandomPersonalUids(bugbanner.BugBannerTask):
    type = 'GET_RANDOM_PERSONAL_UIDS'

    input_parameters = [UidsCountParameter]
    execution_space = 1024
    LOCAL_UIDS = 'active-uids'

    def on_enqueue(self):
        SandboxTask.on_enqueue(self)
        channel.task = self
        resource_descr = 'Personal UIDs (Count: {}, Date: {})'.format(
            self.ctx[UidsCountParameter.name],
            datetime.datetime.now().strftime("%d %b %Y %H:%M")
        )
        resource_name = '{}_yandex_uid_list_{}'.format(
            self.ctx[UidsCountParameter.name],
            datetime.datetime.now().strftime("%d_%m_%Y_%H_%M")
        )
        self.ctx['out_resource_id'] = self.create_resource(
            resource_descr,
            resource_name,
            resource_types.YANDEX_UID_LIST,
            arch='any'
        ).id

    def on_execute(self):
        self.add_bugbanner(bugbanner.Banners.NoapacheUpper, add_responsibles=["aokhotin", "tapot"])

        with VirtualEnvironment(use_system=True, ) as venv:
            env.PipEnvironment(
                "runtime", version="0.1.68", venv=venv, index_url="http://pypi.yandex-team.ru/simple/", use_wheel=True
            ).prepare()
            out_res = channel.sandbox.get_resource(self.ctx['out_resource_id'])
            process.run_process(
                [
                    venv.executable,
                    os.path.join(os.path.dirname(__file__), "dump_random_uids.py"),
                    '--count', str(self.ctx[UidsCountParameter.name]),
                    '--out-file', out_res.path,
                ],
                log_prefix="dump_random_uids",
            )


__Task__ = GetRandomPersonalUids
