# -*- coding: utf-8 -*-

import logging
import os

import sandbox.common.types.client as ctc

from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk import parameters as sp
from sandbox.sandboxsdk import process

from sandbox.projects import resource_types
from sandbox.projects.common import apihelpers
from sandbox.projects.common import utils


class MrServerParameter(sp.SandboxStringParameter):
    name = 'mr_server'
    description = 'MR Server'
    default_value = "hahn"


class ServicesParameter(sp.SandboxBoolGroupParameter):
    name = 'services'
    description = 'Services'
    choices = [
        ('Web', 'web'),
        ('Images', 'images'),
        ('Video', 'video'),
    ]
    default_value = 'web'
    group = 'Services'


class NumOfRequestParameter(sp.SandboxIntegerParameter):
    name = 'num_of_requests'
    description = 'Number of requests'
    default_value = 1000


class RegionalParameter(sp.SandboxSelectParameter):
    name = 'region'
    description = 'Region'
    choices = (
        ("Not specified", 0),
        ("RU", 225),
        ("TR", 983),
        ("BY", 149),
        ("KZ", 159),
        ("UA", 187),
    )
    default_value = 0


class SearchPropFilterParameter(sp.SandboxStringParameter):
    name = 'search_prop_filter'
    description = 'Filter search_prop'
    required = False
    default_value = None


class RandomQueriesBinary(sp.TaskSelector):
    name = 'random_queries_binary_builder'
    description = 'Random queries binary builder'
    task_type = 'BUILD_ARCADIA_BINARY'
    required = False
    default_value = None


class UseLatestRandomQueriesBinary(sp.SandboxBoolParameter):
    name = 'use_latest_random_queries_binary'
    description = 'Get random_queries_binary from somewhere else in sandbox if possible'
    default_value = False


class LogsTypeParameter(sp.SandboxSelectParameter):
    name = 'logs_type'
    description = 'Type of logs to use; '
    choices = (
        ("session", "session"),
        ("reqans", "reqans"),
        ("reqans_proto", "reqans_proto"),
        ("nano", "nano"),
    )
    default_value = "reqans"


class CompleteEarlyParameter(sp.SandboxBoolParameter):
    name = 'complete_early'
    description = 'Complete operation once we have enough queries'
    default_value = True


class GetRandomRequests(SandboxTask):
    type = 'GET_RANDOM_REQUESTS'
    client_tags = ctc.Tag.Group.LINUX
    input_parameters = [
        MrServerParameter,
        ServicesParameter,
        NumOfRequestParameter,
        RegionalParameter,
        SearchPropFilterParameter,
        RandomQueriesBinary,
        UseLatestRandomQueriesBinary,
        LogsTypeParameter,
        CompleteEarlyParameter,
    ]

    def initCtx(self):
        self.ctx['notify_via'] = 'email'

    def on_enqueue(self):
        SandboxTask.on_enqueue(self)

        res_description = 'Results of {} test #{}" ({} random requests)'.format(
            self.type, self.id, self.ctx['num_of_requests'],
        )
        self.ctx['random_queries_resource_id'] = self.create_resource(
            res_description, 'random_queries.txt', resource_types.USERS_QUERIES,
            arch='any',
            attributes={
                'random_requests': self.ctx['num_of_requests'],
                'geo_id': self.ctx["region"] or "no",
            },
        ).id

    def _run_yt_process(self, bundle_path, geodata_path):
        filename = channel.sandbox.get_resource(self.ctx['random_queries_resource_id']).path
        env = os.environ.copy()
        env["MR_RUNTIME"] = "YT"
        env["YT_PROXY"] = utils.get_or_default(self.ctx, MrServerParameter) + ".yt.yandex.net:80"
        env["YT_TOKEN"] = self.get_vault_data("SEARCH-RELEASERS", "yt_token")
        env["YT_SPEC"] = '{"mapper": {"memory_limit": 8589934592}, "data_size_per_job": 17179869184}'

        props = []
        for key, item in (
            ("-r", RegionalParameter.name),
            ("--sp", SearchPropFilterParameter.name),
            ("-l", LogsTypeParameter.name),
        ):
            if item in self.ctx and self.ctx[item]:
                props.append(key)
                props.append(str(self.ctx[item]))
        if self.ctx.get(CompleteEarlyParameter.name):
            props.append('--complete_op_early')

        commands = [
            ("tar -zxf {}".format(bundle_path), "unpack_binary"),
            ("tar -zxf {}".format(geodata_path), "unpack_geodata"),
            (
                "./get_random_queries -c {count} {props} --tables_count 24 "
                "--geodata {geodata} --print --drop_table > {filename}".format(
                    count=self.ctx[NumOfRequestParameter.name],
                    props=" ".join(props),
                    geodata="geodata4.bin",
                    filename=filename,
                ),
                "collect_queries"
            ),
        ]

        for cmd, log_prefix in commands:
            logging.info("Start {part} part of task with command {cmd}".format(part=log_prefix, cmd=cmd))
            process.run_process(
                [cmd], work_dir="./", timeout=3 * 3600, shell=True, check=True,
                log_prefix=log_prefix, outputs_to_one_file=True, environment=env
            )

    def _prepare_task_data(self):
        if self.ctx.get(UseLatestRandomQueriesBinary.name):
            found_tasks = channel.sandbox.list_tasks(
                task_type="BUILD_ARCADIA_BINARY",
                completed_only=True, hidden=True, children=True,
                descr_mask=(
                    "Build get_random_queries binary for "
                    "PREPARE_QUERIES_AND_BINARIES_FOR_REARRANGE_DYNAMIC_TESTS.*"
                )
            )
            logging.info('Total {} tasks found'.format(len(found_tasks)))
            for t in sorted(found_tasks, key=lambda x: int(x.id), reverse=True):
                if t.ctx.get('out_resource_id'):
                    res = channel.sandbox.get_resource(t.ctx['out_resource_id'])
                    if res and res.is_ready():
                        logging.info("Found %s task which built random_queries_binary. Reusing its result", t.id)
                        self.ctx[RandomQueriesBinary.name] = t.id
                        break

        if not self.ctx.get(RandomQueriesBinary.name):
            self.ctx[RandomQueriesBinary.name] = self.create_subtask(
                task_type="BUILD_ARCADIA_BINARY",
                arch="linux",
                description="Build get_random_queries for GetRandomRequest task #{}".format(self.id),
                input_parameters={
                    "notify_if_finished": "",
                    "build_path": "yweb/freshness/get_random_queries",
                    "binaries": "yweb/freshness/get_random_queries/get_random_queries",
                },
            ).id

        utils.check_tasks_to_finish_correctly([self.ctx[RandomQueriesBinary.name]])
        bundle_path = self.sync_resource(
            channel.sandbox.get_task(self.ctx[RandomQueriesBinary.name]).ctx['out_resource_id']
        )
        from projects.geobase.Geodata4BinStable.resource import GEODATA4BIN_STABLE
        geodata_path = self.sync_resource(apihelpers.get_last_released_resource(
            GEODATA4BIN_STABLE
        ).id)

        return bundle_path, geodata_path

    def on_execute(self):
        bundle_path, geodata_path = self._prepare_task_data()
        self._run_yt_process(bundle_path, geodata_path)


__Task__ = GetRandomRequests
