# -*- coding: utf-8 -*-

import sandbox.common.types.client as ctc
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common import utils
from sandbox.projects import resource_types
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk import paths
from sandbox.sandboxsdk import parameters as sp
from sandbox.sandboxsdk import process
import sandbox.projects.common.constants as consts

from sandbox.projects.common.news.routerd import Routerd, create_routerd_params


class AppHostQueries(sp.ResourceSelector):
    name = "apphost_queries"
    description = "Routerd apphost requests"
    resource_type = ['ROUTERD_REQUESTS']
    group = "Routerd requests params"
    requred = True


class AppHostRequester(sp.ResourceSelector):
    name = "apphost_requester"
    description = "Binary to send apphost requests to routerd"
    resource_type = ['SLAVE_NEWSD_APPHOST_REQUESTER']
    group = "Routerd requests params"
    requred = False


# This parameter will be available in "Advanced" section of the task parameters
class ContainerParameter(sp.Container):
    name = consts.SANDBOX_CONTAINER
    description = 'Container the task should execute in'
    default_value = None
    required = False


class GetRouterdResponses(SandboxTask):
    """
        Получает и сохраняет результаты обстрела Routerd apphost'овыми запросами
    """
    type = 'GET_ROUTERD_RESPONSES'

    routerd_params = create_routerd_params()
    input_parameters = \
        routerd_params.params \
        + (AppHostQueries, AppHostRequester, ContainerParameter)
    client_tags = ctc.Tag.Group.LINUX

    def on_enqueue(self):
        SandboxTask.on_enqueue(self)
        resource = self.create_resource(
            self.descr,
            "routerd_responses.txt",
            resource_types.ROUTERD_RESPONSES
        )
        self.ctx['out_resource_id'] = resource.id

    def on_execute(self):
        out_resource = channel.sandbox.get_resource(self.ctx['out_resource_id'])

        workdir = self.path('work')
        paths.make_folder(workdir, True)
        port = 17171

        timestamp = channel.sandbox.get_resource_attribute(
            self.ctx[AppHostQueries.name],
            attribute_name="timestamp",
        )

        routerd = Routerd(
            binary=self.sync_resource(self.ctx[self.routerd_params.Binary.name]),
            workdir=workdir,
            port=port,
            geobase=self.sync_resource(self.ctx[self.routerd_params.Geobase.name]),
            newsdata=self.sync_resource(self.ctx[self.routerd_params.NewsData.name])+"/newsdata2.json",
            newsdata_exp=self.sync_resource(self.ctx[self.routerd_params.NewsDataExp.name])+"/newsdata2.json",
            device_data=self.sync_resource(self.ctx[self.routerd_params.DeviceData.name]),
            allowed_origins=self.sync_resource(self.ctx[self.routerd_params.AllowedOrigins.name]),
            dynamic_robots_txt=self.sync_resource(self.ctx[self.routerd_params.DynamicRobotsTxtConfig.name]),
            now=timestamp,
        )

        if self.ctx.get(AppHostRequester.name):
            requester = self.sync_resource(self.ctx[AppHostRequester.name])
        else:
            resource_id = utils.get_and_check_last_released_resource_id(
                resource_types.SLAVE_NEWSD_APPHOST_REQUESTER,
                arch='linux',
            )
            requester = self.sync_resource(resource_id)
        requests = self.sync_resource(self.ctx[AppHostQueries.name])

        routerd.start()
        routerd.wait()

        cmd = [
            requester,
            '--address', "post://localhost:{}".format(str(port)),
            '--input', str(requests),
            '--output', str(out_resource.path),
        ]
        self.process_parameters = cmd

        self.process = process.run_process(cmd, log_prefix='requester_', wait=True)

        eh.ensure(routerd.is_running(), "Routerd crashed on some request")

        self.mark_resource_ready(self.ctx['out_resource_id'])


__Task__ = GetRouterdResponses
