# -*- coding: utf-8 -*-

import logging

from sandbox import sdk2
from sandbox.sandboxsdk import parameters
import sandbox.common.types.client as ctc

from sandbox.projects.common.BaseGetMediaDatabaseTask import BaseGetVideoDatabaseTask
from sandbox.projects.common import utils
from sandbox.projects import resource_types

from sandbox.projects.common.search.database import iss_shards
from sandbox.projects.common.search.settings import VLA_REGION
from sandbox.projects.common.search.settings import INDEX_MAIN
from sandbox.projects.common.search.settings import VideoSettings
from sandbox.projects.cajuper.resources import VideoProductionIndexStateResource


class GetTier0Parameter(parameters.SandboxBoolParameter):
    name = 'get_tier0_resources'
    description = 'Get resources for Tier0 (or Platinum)'
    default_value = False


class ShardIdParameter(parameters.SandboxStringParameter):
    name = 'shard_id'
    description = 'Shard id like "013"'
    default_value = '009'


def getDbName(resource):
    prefix = 'export YT_STATE='
    for line in open(str(sdk2.ResourceData(sdk2.Resource.find(id=resource.id).first()).path)).readlines():
        line = line.strip()
        if line.startswith(prefix):
            return line[len(prefix):]
    return ''


class GetVideoSearchDatabase(BaseGetVideoDatabaseTask):
    """
        Скачивает базу для видео базового поиска по указанной rsync ссылке
    """
    type = 'GET_VIDEO_SEARCH_DATABASE'
    client_tags = ctc.Tag.Group.LINUX
    execution_space = 40 * 1024  # 40 Gb
    disk_space = 40 * 1024  # 40 Gb

    input_parameters = [GetTier0Parameter, ShardIdParameter] + BaseGetVideoDatabaseTask.input_parameters

    def _get_database_resource_type(self):
        return resource_types.VIDEO_SEARCH_DATABASE

    def _get_expected_files(self):
        return BaseGetVideoDatabaseTask._get_expected_files(self) + ["indexuserurldat"]

    def _get_filter_out_host_patterns(self):
        return []

    def _get_cms_configurations(self):
        cms_configuration, _ = VideoSettings.basesearch_tag(INDEX_MAIN, VLA_REGION, VideoSettings.get_nanny_token(self))
        return [cms_configuration, 'BETA']

    def _get_default_shard_instance_tag(self):
        _, instance_tag_name = VideoSettings.basesearch_tag(INDEX_MAIN, VLA_REGION, VideoSettings.get_nanny_token(self))
        return instance_tag_name

    def _get_stable_basesearch_database_resource(self):
        return utils.get_and_check_last_resource_with_attribute(VideoProductionIndexStateResource,
                                                                attr_name='released',
                                                                attr_value='stable')

    def _get_default_rsync_path(self):
        logging.info("Looking for default shard")

        stable_resource = self._get_stable_basesearch_database_resource()
        date, time = getDbName(stable_resource).split('-')
        shardType = self.ctx[GetTier0Parameter.name] and 'vidsidx' or 'vidsidxpt'
        shardId = self.ctx[ShardIdParameter.name]
        shard_name = '{name}-{shardid}-{date}-{time}'.format(name=shardType, shardid=shardId, date=date, time=time)
        shard_location = iss_shards.get_shard_name(shard_name)
        return shard_location


__Task__ = GetVideoSearchDatabase
