# coding: utf-8

import logging
import os

import sandbox.common.types.resource as ctr
import sandbox.sdk2.environments as env
from sandbox import sdk2
from sandbox.projects.websearch.begemot.resources import WizardQueriesJson
from sandbox.projects import resource_types


class GetWizardProdQueries(sdk2.Task):
    """
        Task for getting wizard/begemot queries from YT
    """
    class Requirements(sdk2.Requirements):
        disk_space = 10 * 1024  # 10 Gb
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):
        day = sdk2.parameters.String("Day")
        query_limit = sdk2.parameters.Integer("Wizard query limit", default=1000)
        add_te_attrs = sdk2.parameters.Bool("Add TE prod attrs", default=False)

    def on_execute(self):
        selected_day = self.Parameters.day.replace("_", "")
        with env.VirtualEnvironment() as venv:
            logging.info('Installing yql')
            env.PipEnvironment('yql', version="1.2.49", venv=venv, use_wheel=True).prepare()
            env.PipEnvironment('yandex-yt', version="0.8.29.post0", venv=venv, use_wheel=True).prepare()
            env.PipEnvironment(
                'yandex-yt-yson-bindings-skynet', version="0.3.7.post1", venv=venv, use_wheel=True
            ).prepare()
            if self.Parameters.query_limit > 0:
                apphost_converter = list(sdk2.Resource.find(
                    resource_type=resource_types.APP_HOST_TOOL_CONVERTER_EXECUTABLE,
                    state=ctr.State.READY,
                    attr_name="released",
                    attr_value="stable",
                ).limit(1))[0]
                output_res = WizardQueriesJson(
                    self, "Wizard prod queries", "wizard_prod_queries_{}.txt".format(selected_day)
                )
                if self.Parameters.add_te_attrs:
                    setattr(output_res, "TE_web_production_wizard_reqs", self.Parameters.day)
                cmd = [
                    venv.executable, os.path.dirname(__file__) + '/select_wizard_queries.py',
                    "--day", selected_day,
                    "--limit", str(self.Parameters.query_limit),
                    "--token", sdk2.Vault.data("BEGEMOT", "Begemot YQL token"),
                    "--qpath", str(sdk2.ResourceData(output_res).path),
                    "--qtype", "begemot_worker_bravo_production_man-requests-log",
                    "--converter", str(sdk2.ResourceData(apphost_converter).path),
                ]
                with sdk2.helpers.ProcessLog(self, logger="select_wizard_queries") as pl:
                    sdk2.helpers.subprocess.check_call(" ".join(cmd), shell=True, stdout=pl.stdout, stderr=pl.stderr)
