"""
    Get production queries form eventlogdata on banach
"""
from __future__ import print_function
from six.moves import range

import argparse
import base64
import os
import subprocess
import time
import traceback


def sslfix():  # YTADMINREQ-8845
    import certifi
    import requests

    YANDEX_SSL_CERTIFICATE = 'https://crls.yandex.net/YandexInternalRootCA.crt'

    ca_bundle = certifi.where()
    resp = requests.get(YANDEX_SSL_CERTIFICATE)
    with open(ca_bundle, 'a') as ca_file:
        ca_file.write('\n# Yandex Internal Root CA\n')
        ca_file.write(resp.text)


DELIMITER = "\n--------\n"


def get_queries(client, day, limit, queries_type, queries_path, converter_path):
    yql_req = (
        'PRAGMA inferscheme;\n'
        'SELECT event_data FROM [home/search-runtime/eventlogdata/{queries_type}/{day}]\n'
        'WHERE event_type == "ReqWizardRequestReceived" AND event_data LIKE "%\tapphost/[%"'
        'LIMIT {limit};'.format(queries_type=queries_type, day=day, limit=limit)
    )
    print("Use YQL query:")
    print(yql_req)
    request = client.query(yql_req)
    request.run()

    allowed_bad_rows = 10
    bad_rows = 0
    with open(queries_path, "w") as f:
        for table in request.get_results():
            n_try = 0
            for n_try in range(1, 4):
                try:
                    table.fetch_full_data()
                    break
                except Exception as e:
                    print("Failed to fetch full data {} time".format(n_try))
                    print(e)
                    time.sleep(20)
            if n_try >= 3:
                raise RuntimeError("Can't fetch full data from banach")
            for row in table.rows:
                if row:
                    stripped_row = row[0].strip()
                    try:
                        decoded_row = base64.b64decode(stripped_row.split("\t")[0])
                        proc = subprocess.Popen(
                            [converter_path, '--input-format', 'service_request'],
                            stdin=subprocess.PIPE, stdout=subprocess.PIPE, stderr=subprocess.PIPE
                        )
                        out, err = proc.communicate(decoded_row)
                        if proc.returncode:
                            raise Exception("Failed to convert: {}".format(traceback.format_exc()))
                        f.write(out + DELIMITER)
                    except StandardError:
                        print("Bad row: '{}', skip it.\nError: {}".format(stripped_row, traceback.format_exc()))
                        bad_rows += 1
                        if bad_rows > allowed_bad_rows:
                            raise RuntimeError("Too many bad rows")

    if os.path.getsize(queries_path) < 5 * 1024 * 1024:  # check number of queries in small files (< 5 Mb)
        with open(queries_path) as f_check:
            n_queries_got = f_check.read().count(DELIMITER)
            if n_queries_got < limit / 2:
                raise RuntimeError("Not enough queries got: {}".format(n_queries_got))


def main():
    from yql.api.v1.client import YqlClient
    pars = argparse.ArgumentParser(description="Get wizard production queries")
    pars.add_argument("--day", "-d", help="Day of eventlog data")
    pars.add_argument("--limit", "-l", type=int, help="Limit of queries")
    pars.add_argument("--qpath", "-p", help="Path to output queries")
    pars.add_argument("--qtype", "-Q", help="Folder name in YT")
    pars.add_argument("--token", "-t", help="Token")
    pars.add_argument("--converter", help="Apphost converter path")
    opts = pars.parse_args()

    sslfix()
    client = YqlClient(db='banach', token=opts.token)
    get_queries(client, opts.day, opts.limit, opts.qtype, opts.qpath, opts.converter)


if __name__ == '__main__':
    main()
