# encoding: utf-8

"""Build hot_water push and card data. Send out pushes.

Uses binary resources created in: https://sandbox.yandex-team.ru/task/119486433/view (copy, check arcadia rev).
Don't forget to add a big ttl to the resource.
"""

import logging
import os

from datetime import date

from sandbox import sdk2
from sandbox.projects import resource_types
from sandbox.projects.common import time_utils as tu
from sandbox.sdk2.helpers import subprocess as sp


class HWCardPushLogResource(sdk2.Resource):
    """Pushes sent with reqids log_file resource."""
    pass


class HWCardsPush(sdk2.Task):
    """Gather hot water push data; send pushes."""

    class Requirements(sdk2.Task.Requirements):
        disk_space = 20 * 1024
        ram = 8 * 1024

    class Parameters(sdk2.Parameters):
        kill_timeout = 83000

        collect_data = sdk2.parameters.Bool(
            'Collect hot water data for pushes and cards.',
            default=True
        )
        send_pushes = sdk2.parameters.Bool(
            'Send hot water pushes from table, filtered by date to send.',
            default=True
        )
        geo_date = sdk2.parameters.String(
            'Geocoded data date. Subsequent tables folder date.',
            default=''
        )
        datasync = sdk2.parameters.Bool(
            'Parse datasync table.',
            default=True
        )
        schedule = sdk2.parameters.Bool(
            'Parse schedule table.',
            default=True
        )
        sup_users = sdk2.parameters.Bool(
            'Cross-reference with sup user data.',
            default=True
        )
        ids = sdk2.parameters.Bool(
            'Join other ids from crypta.',
            default=True
        )
        cards = sdk2.parameters.Bool(
            'Build card and push tables.',
            default=True
        )
        table = sdk2.parameters.String(
            'Table with push data to send.',
            default=''
        )
        dates = sdk2.parameters.String(
            'Push dates to filter by (send date).'
        )

    def on_execute(self):
        cmd_args = []

        # Flags to parse data
        if self.Parameters.collect_data:
            logging.info('\n{time}:\tWill collect hot water data.\n'.format(time=tu.date_ymdhm()))
            cmd_args.append('--collect-data')

        if self.Parameters.geo_date:
            logging.info('\n{time}:\tUsing geocoded date from: {date}.\n'.format(
                time=tu.date_ymdhm(),
                date=self.Parameters.geo_date
            ))
            cmd_args.extend(['--date', self.Parameters.geo_date])

        if self.Parameters.datasync:
            logging.info('\n{time}:\tGetting addresses from datasync.\n'.format(time=tu.date_ymdhm()))
            cmd_args.append('--datasync')

        if self.Parameters.schedule:
            logging.info('\n{time}:\tGetting data with schedule.\n'.format(time=tu.date_ymdhm()))
            cmd_args.append('--schedule')

        if self.Parameters.sup_users:
            logging.info('\n{time}:\tAdding sup users info.\n'.format(time=tu.date_ymdhm()))
            cmd_args.append('--sup')

        if self.Parameters.ids:
            logging.info('\n{time}:\tAdding other ids.\n'.format(time=tu.date_ymdhm()))
            cmd_args.append('--ids')

        if self.Parameters.cards:
            logging.info('\n{time}:\tCreating psuh and card tables.\n'.format(time=tu.date_ymdhm()))
            cmd_args.append('--cards')

        # Flags to send pushes
        if self.Parameters.send_pushes:
            logging.info('\n{time}:\tWill send pushes.\n'.format(time=tu.date_ymdhm()))
            cmd_args.append('--send-push')

        if self.Parameters.table:
            logging.info('\n{time}:\tTable to get pushes from: {table}.\n'.format(
                time=tu.date_ymdhm(),
                table=self.Parameters.table
            ))
            cmd_args.extend(['--table', self.Parameters.table])

        if self.Parameters.dates:
            dates = self.Parameters.dates
            cmd_args.extend(['--dates', dates])
            logging.info('\n{time}:\tDate string filter for pushes: {dates}.\n'.format(
                time=tu.date_ymdhm(),
                dates=dates
            ))

        hotwater_push_binary_resources = sdk2.Resource.find(
            resource_type=resource_types.HW_PUSH_TOOL,
            state='READY'
        ).limit(3)

        last_binary = sorted(list(hotwater_push_binary_resources), key=lambda x: x.created)[-1]
        last_binary_path = str(sdk2.ResourceData(last_binary).path)

        logging.info('Using HWTool binary created on: ' + last_binary.created.strftime('%Y-%m-%d'))
        logging.info('=======\n\n', last_binary_path, cmd_args, '=======\n\n')

        os.environ['PUSH_TOKEN'] = sdk2.Vault.data('antonka_push_token')
        os.environ['YT_TOKEN'] = sdk2.Vault.data('antonka', 'YT_TOKEN')
        os.environ['YT_PROXY'] = 'hahn.yt.yandex.net'

        pushes_sent_log_resource = sdk2.ResourceData(HWCardPushLogResource(
            self,
            'Log resource with pushes sent count and reqids',
            'push_{}.log'.format(date.today().strftime('%Y%m%d')),
            ttl=30
        ))

        with sdk2.helpers.ProcessLog(self, logger="hw_push") as pl:
            sp.Popen([last_binary_path] + cmd_args, stdout=pl.stdout, stderr=sp.STDOUT).wait()

        pushes_sent_log_resource.ready()
