import logging

from sandbox.projects.Haas.common import params
from sandbox.projects.Haas.common import clients
from sandbox.projects.Haas.common import helpers
from sandbox.projects.Haas.common import constants

from sandbox.sandboxsdk import errors

from sandbox import sdk2


class HaasDiscover(sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 60 * 30  # 30 minutes

        with sdk2.parameters.Group('Host discover specification') as host_specs:
            host_filter = params.host_filter
            host_identifier = sdk2.parameters.String('Host identifier', required=True)

        with sdk2.parameters.Group('Discover task settings') as task_settings:
            status_update_period = params.status_update_period
            operations_timeout = params.get_operations_timeout(60 * 30)

        with sdk2.parameters.Group('API specification') as api_urls:
            eine_api_url = params.eine_api_url
            eine_token_name = params.eine_token_name
            eine_token_owner = params.eine_token_owner
            haas_core_api_url = params.haas_core_api_url
            haas_core_token_name = params.haas_core_token_name
            haas_core_token_owner = params.haas_core_token_owner

    def on_execute(self):
        eine_token = sdk2.Vault.data(self.Parameters.eine_token_owner, self.Parameters.eine_token_name)
        eine_client = clients.EineClient(self.Parameters.eine_api_url, eine_token)

        with self.memoize_stage.discover:
            einstellung = _discover_using_eine(
                self.Parameters.host_filter,
                self.Parameters.host_identifier,
                eine_client,
            )
            self._wait_until_discover_done(einstellung, eine_client)

        with self.memoize_stage.push_discover_info:
            network_info = _get_network_info_from_eine(
                self.Parameters.host_filter,
                self.Parameters.host_identifier,
                eine_client,
            )
            core_token = sdk2.Vault.data(self.Parameters.haas_core_token_owner, self.Parameters.haas_core_token_name)
            core_client = clients.HaasCoreClient(self.Parameters.haas_core_api_url, core_token)
            core_client.push_discover_info(self.Parameters.host_identifier, network_info)

    def _wait_until_discover_done(self, einstellung, eine_client):
        logging.info('Waiting until discover finished.')
        helpers.check_until_not_true(
            self.Parameters.status_update_period,
            self.Parameters.operations_timeout,
            eine_client.einstellung_finished,
            einstellung,
        )

        if eine_client.einstellung_failed(einstellung):
            raise errors.SandboxTaskFailureError('Failed during discover "{}"'.format(einstellung.einstellung_id))


def _discover_using_eine(host_filter, identifier, eine_client):
    logging.info('Host to discover: {}'.format(identifier))
    return eine_client.assign_discovery(
        host_filter,
        identifier
    )


def _get_network_info_from_eine(host_filter, identifier, eine_client):
    network_info = eine_client.get_host_properties(
        host_filter,
        identifier,
        [constants.EineHostProperties.switch, constants.EineHostProperties.last_active_mac],
    )
    logging.info('Found network info {}'.format(network_info))
    return network_info
