import time
import logging

from sandbox.projects.Haas.common import params
from sandbox.projects.Haas.common import clients
from sandbox.projects.Haas.common import helpers
from sandbox.projects.Haas.common import constants

from sandbox import sdk2


class HaasReboot(sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 60 * 60

        with sdk2.parameters.Group('Haas task settings') as task_settings:
            haas_task_id = sdk2.parameters.String('Haas task identifier', required=True)
            operations_timeout = params.get_operations_timeout(60 * 5)
            status_update_period = params.status_update_period()

        with sdk2.parameters.Group('Host specification') as host_specs:
            host_filter = params.host_filter()
            host_id = sdk2.parameters.String('Host identifier', required=True)

        with sdk2.parameters.Group('API specification') as api_urls:
            haas_core_api_url = params.haas_core_api_url()
            haas_core_token_name = params.haas_core_token_name()
            haas_core_token_owner = params.haas_core_token_owner()

    class Context(sdk2.Task.Context):
        host_inventory = None

    def on_execute(self):
        core_token = sdk2.Vault.data(self.Parameters.haas_core_token_owner, self.Parameters.haas_core_token_name)
        core_client = clients.HaasCoreClient(self.Parameters.haas_core_api_url, core_token)
        core_client.push_task_status(self.Parameters.haas_task_id, constants.HaasCoreTaskStatuses.running)

        with self.memoize_stage.get_host_info:
            with core_client.step_logging(self.Parameters.haas_task_id, 'GATHERING_HOST_INFO'):
                self._gather_host_info(
                    self.Parameters.host_filter,
                    self.Parameters.host_id,
                    core_client,
                )

        with self.memoize_stage.try_to_reboot:
            self.try_to_reboot(core_client)

    def try_to_reboot(self, core_client):
        try:
            with core_client.step_logging(self.Parameters.haas_task_id, 'FIRST_TRY'):
                self.stop_and_start_server(core_client)
        except helpers.TimeoutError:
            logging.info('First try failed because of timeout, trying BMC reset cold.')
            self.bmc_reset_and_wait(core_client)
            self.second_try(core_client)

    def bmc_reset_and_wait(self, core_client):
        with core_client.step_logging(self.Parameters.haas_task_id, 'BMC_RESET_COLD'):
            core_client.ipmi.bmc_reset_cold(self.Context.host_inventory)
            logging.info('Sleeping {} seconds after BMC reset cold'.format(self.Parameters.operations_timeout))
            time.sleep(self.Parameters.operations_timeout)

    def second_try(self, core_client):
        try:
            with core_client.step_logging(self.Parameters.haas_task_id, 'SECOND_TRY'):
                self.stop_and_start_server(core_client)
        except helpers.TimeoutError:
            logging.info('Second try failed because of timeout, sending reboot ticket.')
            self.send_reboot_ticket(core_client)

    def send_reboot_ticket(self, core_client):
        with core_client.step_logging(self.Parameters.haas_task_id, 'SENDING_REBOOT_TICKET'):
            core_client.send_reboot_ticket(self.Context.host_inventory)
            logging.info('Reboot ticket was sent.')

    def stop_and_start_server(self, core_client):
        with core_client.step_logging(self.Parameters.haas_task_id, 'STOPPING_SERVER'):
            core_client.ipmi.power_off(self.Context.host_inventory)
            self._wait_for_target_power_status(constants.IPMIPowerStatuses.off, core_client)

        with core_client.step_logging(self.Parameters.haas_task_id, 'STARTING_SERVER'):
            core_client.ipmi.power_on(self.Context.host_inventory)
            self._wait_for_target_power_status(constants.IPMIPowerStatuses.on, core_client)

    def _gather_host_info(self, host_filter, host_id, core_client):
        host_info = core_client.get_host_info(host_filter, host_id)
        self.Context.host_inventory = host_info.id

    def _wait_for_target_power_status(self, target_power_status, core_client):
        def power_status_is_target():
            current_power_status = core_client.ipmi.get_power_status(self.Context.host_inventory)
            logging.info('Current power status is "{}", target is "{}"'
                         .format(current_power_status, target_power_status))
            return current_power_status == target_power_status

        helpers.check_until_not_true(
            self.Parameters.status_update_period,
            self.Parameters.operations_timeout,
            power_status_is_target,
        )

    def on_success(self, prev_status):
        core_token = sdk2.Vault.data(self.Parameters.haas_core_token_owner, self.Parameters.haas_core_token_name)
        core_client = clients.HaasCoreClient(self.Parameters.haas_core_api_url, core_token)
        core_client.push_task_status(self.Parameters.haas_task_id, constants.HaasCoreTaskStatuses.success)
        super(HaasReboot, self).on_success(prev_status)

    def on_failure(self, prev_status):
        core_token = sdk2.Vault.data(self.Parameters.haas_core_token_owner, self.Parameters.haas_core_token_name)
        core_client = clients.HaasCoreClient(self.Parameters.haas_core_api_url, core_token)
        core_client.push_task_status(self.Parameters.haas_task_id, constants.HaasCoreTaskStatuses.failed)
        super(HaasReboot, self).on_failure(prev_status)
