import logging

from sandbox.projects.Haas.common import params
from sandbox.projects.Haas.common import clients
from sandbox.projects.Haas.common import helpers
from sandbox.projects.Haas.common import constants
from sandbox.projects.Haas.HaasDiscover import HaasDiscover

from sandbox.sandboxsdk import errors
import sandbox.common.types.task as ctt

from sandbox import sdk2


class HaasRedeploy(sdk2.Task):
    SETUP_VLAN = 'A542'

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 60 * 60 * 3

        with sdk2.parameters.Group('Haas task settings') as task_settings:
            haas_task_id = sdk2.parameters.String('Haas task identifier', required=True)
            port_discover = sdk2.parameters.Bool('Port discover', required=True)
            status_update_period = params.status_update_period()
            operations_timeout = params.get_operations_timeout(60 * 30)

        with sdk2.parameters.Group('Host deploy specification') as host_specs:
            host_filter = params.host_filter()
            host_id = sdk2.parameters.String('Host identifier', required=True)
            deploy_config = sdk2.parameters.String('Configuration', required=True)
            switch_to_setup_vlan = sdk2.parameters.Bool('Switch to setup VLAN', default=True, required=True)
            target_vlan = sdk2.parameters.String('Target VLAN', default=constants.VLANS.common)

        with sdk2.parameters.Group('API specification') as api_urls:
            eine_api_url = params.eine_api_url()
            eine_token_name = params.eine_token_name()
            eine_token_owner = params.eine_token_owner()
            setup_api_url = params.setup_api_url()
            setup_token_name = params.setup_token_name()
            setup_token_owner = params.setup_token_owner()
            haas_core_api_url = params.haas_core_api_url()
            haas_core_token_name = params.haas_core_token_name()
            haas_core_token_owner = params.haas_core_token_owner()

    class Context(sdk2.Task.Context):
        host_inventory = None
        host_name = None
        host_macs = None

    def on_execute(self):
        core_token = sdk2.Vault.data(self.Parameters.haas_core_token_owner, self.Parameters.haas_core_token_name)
        core_client = clients.HaasCoreClient(self.Parameters.haas_core_api_url, core_token)
        core_client.push_task_status(self.Parameters.haas_task_id, constants.HaasCoreTaskStatuses.running)

        with self.memoize_stage.get_host_info:
            with core_client.step_logging(self.Parameters.haas_task_id, 'GATHERING_HOST_INFO'):
                self._gather_host_info(
                    self.Parameters.host_filter,
                    self.Parameters.host_id,
                    core_client,
                )

        with self.memoize_stage.discover:
            if self.Parameters.port_discover:
                core_client.step_logging.push_message_to_log(
                    self.Parameters.haas_task_id,
                    'PORT_DISCOVER',
                    core_client.step_logging.RUNNING,
                )

                self.Context.discover_task_id = self._spawn_discover_task()
                raise sdk2.WaitTask(
                    tasks=self.Context.discover_task_id,
                    statuses=tuple(ctt.Status.Group.FINISH) + tuple(ctt.Status.Group.BREAK),
                    wait_all=True,
                )

        with self.memoize_stage.discover_report:
            if self.Parameters.port_discover:
                child_task = [x for x in self.find(id=self.Context.discover_task_id)][0]
                _process_sb_task_status(
                    child_task.status,
                    self.Parameters.haas_task_id,
                    'PORT_DISCOVER',
                    core_client,
                )

        if self.Parameters.switch_to_setup_vlan:
            with self.memoize_stage.set_setup_vlan:
                with core_client.step_logging(self.Parameters.haas_task_id, 'SETTING_SETUP_VLAN'):
                    core_client.set_host_vlan(self.Context.host_inventory, self.SETUP_VLAN)
                    self._wait_until_port_is_switched(self.SETUP_VLAN, core_client)

        with self.memoize_stage.haas_setup_profile:
            with core_client.step_logging(self.Parameters.haas_task_id, 'HAAS_SETUP_PROFILE'):
                eine_token = sdk2.Vault.data(self.Parameters.eine_token_owner, self.Parameters.eine_token_name)
                eine_client = clients.EineClient(self.Parameters.eine_api_url, eine_token)
                eine_client.assign_haas_setup(
                    self.Parameters.host_filter,
                    self.Parameters.host_id,
                )

        with self.memoize_stage.reboot_to_pxe:
            with core_client.step_logging(self.Parameters.haas_task_id, 'REBOOTING_TO_PXE'):
                core_client.ipmi.reboot_to_pxe(self.Context.host_inventory)

        with self.memoize_stage.deploy:
            with core_client.step_logging(self.Parameters.haas_task_id, 'DEPLOYING'):
                setup_token = sdk2.Vault.data(self.Parameters.setup_token_owner, self.Parameters.setup_token_name)
                setup_client = clients.SetupClient(self.Parameters.setup_api_url, setup_token)
                setup_client.deploy_host(
                    self.Context.host_inventory,
                    self.Parameters.deploy_config,
                    self.Context.host_name,
                    self.Context.host_macs,
                )
                self._wait_until_deploy_finished(setup_client)

        if self.Parameters.switch_to_setup_vlan:
            with self.memoize_stage.set_target_vlan:
                with core_client.step_logging(self.Parameters.haas_task_id, 'SETTING_TARGET_VLAN'):
                    core_client.set_host_vlan(self.Context.host_inventory, self.Parameters.target_vlan)
                    self._wait_until_port_is_switched(self.Parameters.target_vlan, core_client)

    def _spawn_discover_task(self):
        parameters = {
            HaasDiscover.Parameters.host_filter.name: self.Parameters.host_filter,
            HaasDiscover.Parameters.host_identifier.name: self.Parameters.host_id,
            HaasDiscover.Parameters.status_update_period.name: self.Parameters.status_update_period,
            HaasDiscover.Parameters.operations_timeout.name: self.Parameters.operations_timeout,
            HaasDiscover.Parameters.eine_api_url.name: self.Parameters.eine_api_url,
            HaasDiscover.Parameters.eine_token_name.name: self.Parameters.eine_token_name,
            HaasDiscover.Parameters.eine_token_owner.name: self.Parameters.eine_token_owner,
            HaasDiscover.Parameters.haas_core_api_url.name: self.Parameters.haas_core_api_url,
            HaasDiscover.Parameters.haas_core_token_name.name: self.Parameters.haas_core_token_name,
            HaasDiscover.Parameters.haas_core_token_owner.name: self.Parameters.haas_core_token_owner,
        }
        discover_task = HaasDiscover(
            self,
            description='Discovering switch and port, using https://eine.yandex-team.ru',
            owner=self.owner,
            **parameters
        )
        discover_task.enqueue()
        return discover_task.id

    def _gather_host_info(self, host_filter, host_id, core_client):
        host_info = core_client.get_host_info(host_filter, host_id)
        self.Context.host_macs = host_info.macs
        self.Context.host_name = host_info.fqdn
        self.Context.host_inventory = host_info.id

    def on_success(self, prev_status):
        core_token = sdk2.Vault.data(self.Parameters.haas_core_token_owner, self.Parameters.haas_core_token_name)
        core_client = clients.HaasCoreClient(self.Parameters.haas_core_api_url, core_token)
        core_client.push_task_status(self.Parameters.haas_task_id, constants.HaasCoreTaskStatuses.success)
        super(HaasRedeploy, self).on_success(prev_status)

    def on_failure(self, prev_status):
        core_token = sdk2.Vault.data(self.Parameters.haas_core_token_owner, self.Parameters.haas_core_token_name)
        core_client = clients.HaasCoreClient(self.Parameters.haas_core_api_url, core_token)
        core_client.push_task_status(self.Parameters.haas_task_id, constants.HaasCoreTaskStatuses.failed)
        super(HaasRedeploy, self).on_failure(prev_status)

    def _wait_until_port_is_switched(self, target_vlan, core_client):
        def check_port_vlan():
            current_vlan = core_client.get_host_vlan(self.Context.host_inventory)
            logging.debug('Current VLAN is "{}", target VLAN is "{}"'.format(
                current_vlan, target_vlan)
            )
            if current_vlan == target_vlan:
                logging.info('Port is switched to VLAN "{}".'.format(target_vlan))
                return True

        helpers.check_until_not_true(
            self.Parameters.status_update_period,
            self.Parameters.operations_timeout,
            check_port_vlan,
        )

    def _wait_until_deploy_finished(self, setup_client):
        def check_deploy_status():
            return setup_client.get_deploy_status(self.Context.host_inventory) in constants.SetupState.final_states

        helpers.check_until_not_true(
            self.Parameters.status_update_period,
            self.Parameters.operations_timeout,
            check_deploy_status,
        )

        if setup_client.get_deploy_status(self.Context.host_inventory) == constants.SetupState.failed:
            raise errors.SandboxTaskFailureError('Deploy was failed.')


def _process_sb_task_status(sb_task_status, haas_task_id, stage, core_client):
    if sb_task_status in ctt.Status.Group.SUCCEED:
        core_client.step_logging.push_message_to_log(
            haas_task_id,
            stage,
            core_client.step_logging.SUCCESS,
        )
    else:
        core_client.step_logging.push_message_to_log(
            haas_task_id,
            stage,
            core_client.step_logging.FAILED,
        )
        raise errors.SandboxTaskFailureError('Child task is failed.')
