import logging

from sandbox.projects.Haas.common import http as haas_http, constants
from .ipmi import IPMICommandsInterface
from .stage_logging import StepLogging
from sandbox.sandboxsdk import errors


class HostInfo(object):
    def __init__(self, host_info):
        for key, value in host_info.iteritems():
            setattr(self, key, value)


class HaasCoreClient(object):
    def __init__(self, api_url, oauth_token):
        self._API_URL = api_url
        self._AUTH_HEADERS = {'Authorization': oauth_token}
        self.ipmi = IPMICommandsInterface(self._API_URL, self._AUTH_HEADERS)
        self.step_logging = StepLogging(self._API_URL, self._AUTH_HEADERS)

    def set_host_vlan(self, host_identifier, target_vlan):
        url = '{}/nodes/{}/net/setup-vlan'.format(self._API_URL, host_identifier)
        data = {'vlans': target_vlan}
        logging.info('Setting host {} VLAN to "{}"'.format(host_identifier, target_vlan))
        haas_http.put(url, data, headers=self._AUTH_HEADERS)

    def get_host_vlan(self, host_identifier):
        url = '{}/nodes/{}/net/setup-vlan'.format(self._API_URL, host_identifier)
        logging.info('Getting host {} current VLAN'.format(host_identifier))
        return haas_http.get(url, headers=self._AUTH_HEADERS)['vlan']['vlans']

    def push_discover_info(self, host_identifier, discover_info):
        url = '{}/nodes/{}/net/switch-port'.format(self._API_URL, host_identifier)
        switch, port = discover_info[constants.EineHostProperties.switch].split()
        data = {
            'mac': discover_info[constants.EineHostProperties.last_active_mac],
            'switch': switch,
            'port': port,
        }
        logging.info('Pushing discover info about {}: "{}"'.format(host_identifier, data))
        haas_http.put(url, data, headers=self._AUTH_HEADERS)

    def push_task_status(self, task_identifier, task_status):
        if task_status not in constants.HaasCoreTaskStatuses.CORRECT:
            message = 'Incorrect task status "{}", you must choose one of "{}"'\
                      .format(task_status, constants.HaasCoreTaskStatuses.CORRECT)
            raise errors.SandboxTaskFailureError(message)

        url = '{}/tasks/{}/status'.format(self._API_URL, task_identifier)
        data = {
            'status': task_status,
        }
        logging.info('Pushing task "{}" status "{}"'.format(task_identifier, task_status))
        haas_http.put(url, data, headers=self._AUTH_HEADERS)

    def send_reboot_ticket(self, host_identifier):
        url = '{}/tickets'.format(self._API_URL)
        data = {
            'type': 'reboot',
            'node_id': host_identifier,
        }
        logging.info('Sending reboot ticket for host "{}"'.format(host_identifier))
        haas_http.post(url, data, headers=self._AUTH_HEADERS)

    def get_host_info(self, host_filter, host_identifier):
        if host_filter not in constants.HaasCoreHostFilters.CORRECT:
            raise errors.SandboxTaskFailureError(
                'Invalid host_filter "{}", you must choose one of "{}"'.format(
                    host_filter,
                    constants.HaasCoreHostFilters.CORRECT,
                )
            )

        logging.info('Getting host info by identifier "{}"'.format(host_identifier))
        if host_filter == constants.HaasCoreHostFilters.inventory:
            url = '{}/nodes/{}'.format(self._API_URL, host_identifier)
            return HostInfo(haas_http.get(url, headers=self._AUTH_HEADERS))

        elif host_filter == constants.HaasCoreHostFilters.hostname:
            url = '{}/nodes?fqdn={}'.format(self._API_URL, host_identifier)
            return HostInfo(haas_http.get(url, headers=self._AUTH_HEADERS)['results'][0])

        else:
            raise errors.SandboxTaskFailureError('Unknown host_filter "{}"'.format(host_filter))
