import logging

from sandbox.projects.Haas.common import http as haas_http
from sandbox.sandboxsdk import errors


class IPMICommandsInterface(object):
    """
    This class generates itself methods, using HaasCoreIPMICommands.ALL as source of data.
    Typical method call looks like this: core_client.ipmi.power_on(host_id), where "ipmi" is IPMICommandsInterface
    instance, "power_on" is HaasCoreIPMICommands.power_on.name.
    """
    IPMI_API = '{}/nodes/{}/ipmi/{}'

    def __init__(self, api_url, auth_headers):
        self._API_URL = api_url
        self._AUTH_HEADERS = auth_headers
        self._ipmi_commands = HaasCoreIPMICommands
        self._generate_ipmi_methods()

    def _generate_ipmi_methods(self):
        for command in self._ipmi_commands.ALL:
            setattr(self, command.method_name, self._generate_method_func(command))

    def _generate_method_func(self, command):
        # Hack to avoid late binding http://docs.python-guide.org/en/latest/writing/gotchas/#id2
        def command_func(inventory):
            return self._do_ipmi_request(inventory, command.api_address, command.http_method, command.data)
        return command_func

    def reboot_to_pxe(self, inventory):
        logging.info('Rebooting host "{}" to PXE'.format(inventory))
        self._call_generated_method(self._ipmi_commands.bootdev_pxe.method_name, inventory)
        self._call_generated_method(self._ipmi_commands.reboot.method_name, inventory)
        logging.info('Host "{}" reboot to PXE succeeded.'.format(inventory))

    def get_power_status(self, inventory):
        logging.debug('Getting power status for host "{}"'.format(inventory))
        status = self._call_generated_method(self._ipmi_commands.power_status.method_name, inventory)
        power_status = status['results']['data']['System Power']
        return power_status

    def _call_generated_method(self, method_name, inventory):
        method = getattr(self, method_name)
        logging.debug('Calling generated method "{}" for host "{}"'.format(method_name, inventory))
        return method(inventory)

    def _do_ipmi_request(self, inventory, api_address, http_method, command_data):
        logging.info('Processing IPMI command "{}" on host "{}".'.format(api_address, inventory))
        url = self.IPMI_API.format(self._API_URL, inventory, api_address)
        response = http_method(url, data=command_data, headers=self._AUTH_HEADERS)
        self._check_response(inventory, response)
        logging.info('IPMI command "{}" on host "{}" succeeded.'.format(api_address, inventory))
        return response

    @staticmethod
    def _check_response(inventory, response):
        logging.debug('Checking operation response "{}"'.format(response))
        if response.get('results') is not None:
            logging.debug('Operation response is ok, operation succeeded.')
        else:
            logging.error('Operation response is not ok, operation failed.')
            raise errors.SandboxTaskFailureError('Host "{}" IPMI command failed.'.format(inventory))


class HaasCoreIPMICommands:
    """
    Make sure that your new command included into ALL in order to IPMICommandsInterface can wrap it into method.
    """

    class IPMICommand:
        def __init__(self, method_name, api_address, http_method, data=None):
            self.method_name = method_name
            self.api_address = api_address
            self.http_method = http_method
            self.data = data

    reboot = IPMICommand('reboot', 'reboot', haas_http.post)
    power_on = IPMICommand('power_on', 'poweron', haas_http.post)
    power_off = IPMICommand('power_off', 'poweroff', haas_http.post)
    power_status = IPMICommand('power_status', 'status', haas_http.get)
    power_commands = {
        reboot,
        power_on,
        power_off,
        power_status,
    }

    bmc_reset_cold = IPMICommand('bmc_reset_cold', 'bmc/reset', haas_http.post, {'mode': 'cold'})
    bmc_reset_warm = IPMICommand('bmc_reset_warm', 'bmc/reset', haas_http.post, {'mode': 'warm'})
    bmc_commands = {
        bmc_reset_cold,
        bmc_reset_warm,
    }

    bootdev_pxe = IPMICommand('bootdev_pxe', 'bootdev/pxe', haas_http.post)
    bootdev_disk = IPMICommand('bootdev_disk', 'bootdev/disk', haas_http.post)
    bootdev_safe = IPMICommand('bootdev_safe', 'bootdev/safe', haas_http.post)
    bootdev_cdrom = IPMICommand('bootdev_cdrom', 'bootdev/cdrom', haas_http.post)
    bootdev_floppy = IPMICommand('bootdev_floppy', 'bootdev/floppy', haas_http.post)
    bootdev_bios = IPMICommand('bootdev_bios', 'bootdev/bios', haas_http.post)
    bootdev_diag = IPMICommand('bootdev_diag', 'bootdev/diag', haas_http.post)
    bootdev_commands = {
        bootdev_pxe,
        bootdev_disk,
        bootdev_safe,
        bootdev_cdrom,
        bootdev_floppy,
        bootdev_bios,
        bootdev_diag,
    }

    ALL = power_commands | bmc_commands | bootdev_commands
