import logging

from sandbox.projects.Haas.common import http as haas_http
from sandbox.projects.Haas.common import constants


class Einstellung(object):
    def __init__(self, host_filter, pattern, einstellung_id):
        self.host_filter = host_filter
        self.pattern = pattern
        self.einstellung_id = einstellung_id

    def __repr__(self):
        return 'Einstellung({}, {}, {})'.format(
            self.host_filter,
            self.pattern,
            self.einstellung_id
        )


class EineClient(object):
    def __init__(self, api_url, oauth_token):
        self._API_URL = api_url
        self._AUTH_HEADERS = {'Authorization': 'OAuth {}'.format(oauth_token)}

    def _assign_profile(self, host_filter, pattern, action, profile):
        data = {'action': action, 'profile': profile}
        url = '{}/computers/{}/{}/einstellungs'.format(self._API_URL, host_filter, pattern)
        eine_id = haas_http.post(url, data, self._AUTH_HEADERS, json=False)['result']['_id']
        return Einstellung(host_filter, pattern, eine_id)

    def _get_einstellung_status(self, einstellung):
        logging.debug('Getting {} status.'.format(einstellung))
        url = '{}/computers/{}/{}/einstellungs/{}'.format(
            self._API_URL,
            einstellung.host_filter,
            einstellung.pattern,
            einstellung.einstellung_id,
        )
        status = haas_http.get(url, headers=self._AUTH_HEADERS)['result']['status']
        logging.debug('Current discover status: {}'.format(status))
        return status

    def _get_host_info(self, host_filter, pattern):
        url = '{}/computers/{}/{}/props'.format(self._API_URL, host_filter, pattern)
        return haas_http.get(url, headers=self._AUTH_HEADERS)['result']

    def get_host_properties(self, host_filter, pattern, properties=()):
        host_info = self._get_host_info(host_filter, pattern)
        return {prop: host_info[prop]['value'] for prop in properties}

    def assign_discovery(self, host_filter, pattern):
        logging.info('Assigning "discovery" profile on host "{}"'.format(pattern))
        return self._assign_profile(host_filter, pattern, 'assign', 'discovery')

    def assign_haas_setup(self, host_filter, pattern):
        logging.info('Assigning "haas-setup" profile on host "{}"'.format(pattern))
        return self._assign_profile(host_filter, pattern, 'assign', 'haas-setup')

    def einstellung_finished(self, einstellung):
        return self._get_einstellung_status(einstellung) in constants.EinstellungStatus.final_states

    def einstellung_failed(self, einstellung):
        return self._get_einstellung_status(einstellung) in constants.EinstellungStatus.failed
