# -*- coding: utf-8 -*-
import logging
import sandbox.common.types.task as ctt
from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.projects.yql.RunYQL2 import RunYQL2

DEFAULT_YQL_TOKEN_VAULT_NAME = 'YQL_TOKEN'


class HaasCmisUpdateDcGrowth(sdk2.Task):
    """ Run me to update 3y growth tables. """

    class Parameters(sdk2.Parameters):
        yql_token_vault_name = sdk2.parameters.String('YQL Token secret name', required=True,
                                                      default=DEFAULT_YQL_TOKEN_VAULT_NAME)
        revision = sdk2.parameters.Integer('Revision of YQL libraries to import', required=True)

    def on_execute(self):
        logging.info('Hello, Sandbox!')

        self._run_stage(stage_name='update_hws_resources_history', stage_callable=self._update_hws_resources_history)
        self._run_stage(stage_name='update_growth_model', stage_callable=self._update_growth_model)
        self._run_stage(stage_name='update_growth_dashboard', stage_callable=self._update_growth_dashboard)

    def _run_stage(self, stage_name, stage_callable):
        context_subtask_id_attr = '{}_task_id'.format(stage_name)
        with getattr(self.memoize_stage, stage_name):
            logging.info('Run task {}'.format(stage_name))
            sub_task_id = stage_callable()
            setattr(self.Context, context_subtask_id_attr, sub_task_id)
            logging.info('Wait output from task {}'.format(sub_task_id))
            raise sdk2.WaitTask(sub_task_id, ctt.Status.Group.FINISH + ctt.Status.Group.BREAK, wait_all=True,
                                timeout=7200)

        sub_task_id = getattr(self.Context, context_subtask_id_attr)
        sub_task = sdk2.Task.find(id=sub_task_id, children=True, status=ctt.Status.SUCCESS) \
            .order(-sdk2.Task.id).first()
        if not sub_task:
            raise TaskFailure('Child task ({}) in bad state: {}'.format(stage_name, sub_task_id))

    def _update_hws_resources_history(self):
        query = """
USE hahn;
PRAGMA yt.InferSchema('1000');
PRAGMA AnsiInForEmptyOrNullableItemsCollections;
PRAGMA library("regular__3y_model.sql");
IMPORT regular__3y_model
SYMBOLS $update_hws_resources_history;

DO $update_hws_resources_history();
        """
        add_files = self._add_3y_model_library()
        sub_task = self._create_yql_sub_task(query=query, add_files=add_files, descr='update_hws_resources_history')
        sub_task.save().enqueue()

        return sub_task.id

    def _update_growth_model(self):
        query = """
USE hahn;
PRAGMA yt.InferSchema('1000');
PRAGMA AnsiInForEmptyOrNullableItemsCollections;
PRAGMA library("regular__3y_model.sql");
IMPORT regular__3y_model
SYMBOLS
    $update_growth_all_dcs,
    $update_growth_by_dc;

DO $update_growth_all_dcs(null);
DO $update_growth_by_dc(null);
        """
        add_files = self._add_3y_model_library()
        sub_task = self._create_yql_sub_task(query=query, add_files=add_files, descr='update_growth_model')
        sub_task.save().enqueue()

        return sub_task.id

    def _update_growth_dashboard(self):
        query = """
USE hahn;
PRAGMA yt.InferSchema('1000');
PRAGMA AnsiInForEmptyOrNullableItemsCollections;
PRAGMA library("regular__3y_model.sql");
IMPORT regular__3y_model
SYMBOLS
    $update_services_to_subgroups_history,
    $update_subgroups_growth_dashboard;

DO $update_services_to_subgroups_history();
COMMIT;
DO $update_subgroups_growth_dashboard();
        """
        add_files = self._add_3y_model_library()
        sub_task = self._create_yql_sub_task(query=query, add_files=add_files, descr='update_growth_dashboard')
        sub_task.save().enqueue()

        return sub_task.id

    def _create_yql_sub_task(self, query, add_files, descr):
        sub_task = RunYQL2(
            self,
            description='Child of task {}: {}'.format(self.id, descr),
            notifications=self.Parameters.notifications,
            create_sub_task=True,
            **{
                RunYQL2.Parameters.query.name: query,
                RunYQL2.Parameters.add_files.name: add_files,
                RunYQL2.Parameters.trace_query.name: True,
                RunYQL2.Parameters.owner.name: self.Parameters.owner,
                RunYQL2.Parameters.publish_query.name: True,
                RunYQL2.Parameters.use_v1_syntax.name: True,
                RunYQL2.Parameters.retry_period.name: 30,
                RunYQL2.Parameters.yql_token_vault_name.name: self.Parameters.yql_token_vault_name
            }
        )
        sub_task.Requirements.ram = 1024

        return sub_task

    def _add_3y_model_library(self):
        return {
            'regular__3y_model.sql':
                'url:arc://haas/cmis/yql/regular__3y_model.sql?rev={}'.format(self.Parameters.revision)
        }
